/* 
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *      author: Mike Henderson mhender@watson.ibm.com
 *      date:   August 26, 1999
 */

static char *id="@(#) $Id: MFPeitgenSystem.c 257 2006-08-30 20:30:07Z mhender $";

#include <MFImplicitMF.h>
#include <MFImplicitMF.h>
#include <MFInvariantMF.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <MFPrint.h>

static char MFPeitgenSystemErrorMsg[256]="";
void MFSetError(int,char*,char*,int,char*);

void MFEvaluatePeitgenSystem(int,MFNVector,MFNVector,void*);
void MFApplyJacobianPeitgenSystem(int,int,MFNVector,MFNKMatrix,MFNKMatrix,void*);
void MFFreePeitgenSystemData(void*);
void MFWritePeitgenSystemData(FILE*,void*);
MFImplicitMF MFReadPeitgenSystem(FILE*);

void MFEvaluatePeitgenSystemRaw(int,double*,double*,void*);

struct MFPeitgenSystemData
 {
  double h;
 };

MFImplicitMF MFCreatePeitgenSystem(double h)
 {
  static char RoutineName[]={"MFCreatePeitgenSystem"};
  MFImplicitMF this;
  MFNSpace space;
  struct MFPeitgenSystemData *data;
  int i,j;

  this=MFIMFCreateBaseClass(2,0,"PeitgenSystem");

  space=MFCreateNSpace(2);
  MFIMFSetSpace(this,space);
  MFFreeNSpace(space);

  data=malloc(sizeof(struct MFPeitgenSystemData)); /*done*/
  if(data==(void*)NULL)
   {
    sprintf(MFPeitgenSystemErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFPeitgenSystemData));
    printf("%s -- %s\n",RoutineName,MFPeitgenSystemErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPeitgenSystemErrorMsg,__LINE__,__FILE__);
    free(this);
    return (MFImplicitMF)NULL;
   }

  data->h=h;

  MFIMFSetFreeData(this,MFFreePeitgenSystemData);
  MFIMFSetEvaluate(this,MFEvaluatePeitgenSystem);
  MFIMFSetApplyJacobian(this,MFApplyJacobianPeitgenSystem);
  MFIMFSetWriteData(this,MFWritePeitgenSystemData);

  return this;
 }

void MFFreePeitgenSystemData(void *d)
 {
  static char RoutineName[]={"MFFreePeitgenSystemData"};
  struct MFPeitgenSystemData *data;
  int i,n;

  data=(struct MFPeitgenSystemData*)d;
  free(data);

  return;
 }

void MFWritePeitgenSystemData(FILE *fid,void *d)
 {
  static char RoutineName[]={"MFWritePeitgenSystemData"};
  struct MFPeitgenSystemData *data;

  data=(struct MFPeitgenSystemData*)d;

  fprintf(fid,"%lf\n",data->h);
  fflush(fid);
  return;
 }

MFImplicitMF MFReadPeitgenSystem(FILE *fid)
 {
  static char RoutineName[]={"MFReadPeitgenSystem"};
  MFImplicitMF this;
  double h=.1;

  fscanf(fid,"%lf\n",&h);

  this=MFCreatePeitgenSystem(h);

  return this;
 }

#define MFPI 3.14159265358979323846264338327950288
#define MFTWOPI 6.2831853071795862320

double PeitgenF(double s)
 {
  static double as;
  static double ss;
  static double result;

  as=fabs(s);

  if(as<MFPI)
    result=sin(s);
   else if(as<MFTWOPI)
    {
     ss=sin(s);
     result=ss-10*ss*ss;
   }else
    result=s-MFTWOPI;

  return result;
 }

double PeitgenDF(double s)
 {
  static double as;
  static double result;

  as=fabs(s);

  if(as<MFPI)
    result=cos(s);
   else if(as<MFTWOPI)
    result=cos(s)*(1-20*sin(s));
   else
    result=1;

  return result;
 }

void MFEvaluatePeitgenSystem(int n,MFNVector vu,MFNVector vf,void *d)
 {
  static char RoutineName[]={"MFEvaluatePeitgenSystem"};
  double *u,*f;

  u=MFNV_CStar(vu);
  f=MFNV_CStar(vf);
  MFEvaluatePeitgenSystemRaw(n,u,f,d);

  return;
 }


void MFEvaluatePeitgenSystemRaw(int n,double *u,double *f,void *d)
 {
  static char RoutineName[]={"MFEvaluatePeitgenSystemRaw"};
  int i;
  struct MFPeitgenSystemData *data;
  static double h;
  static double det;
  static double r0,r1;
  static double a00,a01,a10,a11;
  int verbose;

  verbose=0;
  data=(struct MFPeitgenSystemData*)d;
  h=data->h;

  f[0]=u[0]+h*PeitgenF(u[1]);
  f[1]=u[1]-h*PeitgenF(u[0]);

  r0=f[0]-u[0]+h*PeitgenF((f[1]+u[1])/2);
  r1=f[1]-u[1]-h*PeitgenF((f[0]+u[0])/2);

  if(verbose)printf("%s\n",RoutineName);
  i=0;
  while(fabs(r0)+fabs(r1)>2.e-13)
   {
    if(verbose){printf("  %d %le\n",i,fabs(r0)+fabs(r1));fflush(stdout);}
    a00=1.;
    a01= h*PeitgenDF((f[1]+u[1])/2)/2;
    a01=-h*PeitgenDF((f[0]+u[0])/2)/2;
    a11=1.;
    det=1./(a00*a11-a01*a10);
    f[0]=f[0]-det*( a11*r0-a01*r1);
    f[1]=f[1]-det*(-a10*r0+a00*r1);

    r0=f[0]-u[0]+h*PeitgenF((f[1]+u[1])/2);
    r1=f[1]-u[1]-h*PeitgenF((f[0]+u[0])/2);
   }

  return;
 }

void MFApplyJacobianPeitgenSystem(int m,int k,MFNVector vu, MFNKMatrix mb,MFNKMatrix mc,void *d)
 {
  static char RoutineName[]={"MFApplyJacobianPeitgenSystem"};
  int i,j,l;
  int n;
  struct MFPeitgenSystemData *data;
  static double a[4];
  static double x[2];
  static double Fx,Fy;
  static double h;
  static double det;
  double *u,*b,*c;

/* c = Fu b */

  u=MFNV_CStar(vu);
  b=MFNKM_CStar(mb);
  c=MFNKM_CStar(mc);

  data=(struct MFPeitgenSystemData*)d;
  h=data->h;

  MFEvaluatePeitgenSystemRaw(2,u,x,d);
  Fx=h*PeitgenDF((x[0]+u[0])/2)/2;
  Fy=h*PeitgenDF((x[1]+u[1])/2)/2;

  det=1./(1+Fx*Fy);
  a[0]=det*(1-Fy*Fx);
  a[1]= 2*Fx*det;
  a[2]=-2*Fy*det;
  a[3]=det*(1-Fx*Fy);

  for(i=0;i<2;i++)
    for(j=0;j<k;j++)
      c[i+k*j]=a[i+2*0]*b[0+k*j]+a[i+2*1]*b[1+k*j];

  return;
 }
