/* 
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *      author: Mike Henderson mhender@watson.ibm.com
 */

static char *id="@(#) $Id: MFNewPolytope.c 257 2006-08-30 20:30:07Z mhender $";

#include <math.h>
#include <MFKVector.h>
#include <MFPolytope.h>
#include <MFPrint.h>
#include <MFErrorHandler.h>
#include <stdlib.h>
#include <string.h>
#include <MFTime.h>

/* Internal */

static int *MFPolytopeVertexIndexSet(MFPolytope,int);
static int MFPolytopeVertexLargestIndex(MFPolytope,int);
static int MFPolytopeVertexSmallestIndex(MFPolytope,int);
static double MFPolytopeVertexLargestVertexRadius(MFPolytope);
static int MFPolytopeVertexNumberOfIndices(MFPolytope,int);
static int MFPolytopeLargestVertexIndex(MFPolytope);
static int MFPolytopeSmallestVertexIndex(MFPolytope);
static int MFPolytopeIntersectIndexSets(MFPolytope,int,int,int*);
static int MFPolytopeUnionOfIndexSets(MFPolytope,int,int,int*);
static int MFPolytopeVerticesOnSameEdge(MFPolytope,int,int);
static int MFPolytopeAddVertex(MFPolytope);
static int MFPolytopeAddVertexIndexIntoSet(MFPolytope,int,int);

static char MFPolytopeErrorMsg[256]="";

struct MFPolytopeSt{
                int k;                /* Dimension of the vertices */
                double R;             /* Radius */
                int nv;                /* Number of vertices */
                int mv;                /* Space for vertices */
                double *v;            /* vertices */
                int *nvIndices;        /* Number of indices for each vertex */
                int *mvIndices;        /* Space for indices for each vertex */
                int **vindices;        /* Indices of each vertex */

                int ne;                /* Number of edges */
                int me;                /* Space for edges */
                int *e;                /* edges */
                int *neIndices;        /* Number of indices for each edge */
                int *meIndices;        /* Space for indices for each edge */
                int **eindices;        /* Indices of each edge */
#ifdef STOREFACELIST
                int nFaces;
                int mFaces;
                int *face;          /* Index of face */
                int *nFaceV;        /* Number of vertices on each face */
                MFKVector *faceN;   /* Normal of face */
                double *faceO;      /* Origin of face */
#endif
               };

#define MFPolytopeIncrementToAllocateToVertexList 10
#define MFPolytopeIncrementToAllocateToEdgeList 10
#define MFPolytopeIncrementToAllocateToIndexSet 2
#define MFPolytopeIncrementToAllocateToFaceList 2

MFPolytope MFCreateHyperCubeAtOrigin(int k, double R)
 {
  static char RoutineName[]={"MFCreateHyperCubeAtOrigin"};
  int i,j,l;
  double *c;
  MFPolytope result;
  int carry;
  int verbose;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFCreateHyperCubeAtOrigin++;
  starttime=clock();
#endif

  verbose=0;
  if(verbose)printf("%s\n",RoutineName);fflush(stdout);

/* Allocate storage */

  result=malloc(sizeof(struct MFPolytopeSt)); /*done*/
  if(result==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFPolytopeSt));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }
  if(verbose)printf("MFCreateHyperCubeAtOrigin 0x%8.8x\n",result);

  result->k=k;
  
  result->n=1;for(i=0;i<k;i++)result->n*=2;
  result->m=result->n;
  result->v=malloc(result->n*k*sizeof(double)); /*done*/
  if(result->v==(double*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*k*sizeof(double));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }
  for(i=0;i<result->n*k;i++)result->v[i]=0.;

  result->nIndices=malloc(result->n*sizeof(int)); /*done*/
  if(result->nIndices==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  result->mIndices=malloc(result->n*sizeof(int)); /*done*/
  if(result->mIndices==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  result->indices=malloc(result->n*sizeof(int*)); /*done*/
  if(result->indices==(int**)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*sizeof(int*));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }
  if(verbose)printf("   result(0x%8.8x)->indices=0x%8.8x, %d int*s\n",result,result->indices,result->n);
  for(i=0;i<result->n;i++)
   {
    result->nIndices[i]=0;
    result->mIndices[i]=k;
    result->indices[i]=malloc(k*sizeof(int)); /*done*/
    if(result->indices[i]==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",k*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
      return (MFPolytope)NULL;
     }
    if(verbose)printf("   result(0x%8.8x)->indices[%d]=0x%8.8x, %d ints\n",result,i,result->indices[i],k);
   }

  c=malloc(k*sizeof(double)); /*done*/
  if(c==(double*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",k*sizeof(double));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  for(j=0;j<k;j++)c[j]=-R;

  for(i=0;i<result->n;i++)
   {
    l=i*k;
    if(verbose)printf("vertex %d\n",i);
    for(j=0;j<k;j++)result->v[j+l]=c[j];
    if(verbose){printf(" v=[");for(j=0;j<k;j++){if(j>0)printf(",");printf("%lf",result->v[j+l]);}printf("]\n");}

    result->nIndices[i]=k;
    result->mIndices[i]=k;
    for(j=0;j<k;j++)
     {
      if(c[j]<0)(result->indices[i])[j]=2*j;
       else (result->indices[i])[j]=2*j+1;
     }
    if(verbose){printf(" i=[");for(j=0;j<k;j++){if(j>0)printf(",");printf("%d",(result->indices[i])[j]);}printf("]\n");}

    carry=1;
    j=0;
    while(carry&&j<k)
     {
      if(c[j]<0)
       {
        c[j]=R;
        carry=0;
       }else{
        c[j]=-R;
        carry=1;
        j++;
       }
     }
   }

  if(c==NULL)printf("freeing(0x0) line %d of file %s\n",__LINE__,__FILE__);
  free(c);

  result->nFaces=2*k;
  result->mFaces=result->nFaces;
  result->face=malloc(result->mFaces*sizeof(int)); /*done*/
  if(result->face==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  result->nFaceV=malloc(result->mFaces*sizeof(int)); /*done*/
  if(result->nFaceV==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  result->faceN=malloc(result->mFaces*sizeof(MFKVector)); /*done*/
  if(result->faceN==(MFKVector*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(MFKVector));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  result->faceO=malloc(result->mFaces*sizeof(double)); /*done*/
  if(result->faceO==(double*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(double));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  for(i=0;i<2*k;i++)
   {
    result->face[i]=i;
    result->nFaceV[i]=0;
    result->faceN[i]=MFCreateKVector(k);
    if(i%2==0)
      MFKVSetC(result->faceN[i],i/2,R);
     else
      MFKVSetC(result->faceN[i],i/2,-R);
    result->faceO[i]=1.;
   }

  for(i=0;i<result->n;i++)
   {
    for(j=0;j<result->nIndices[i];j++)
     {
      result->nFaceV[(result->indices[i])[j]]++;
     }
   }

  result->R=R*sqrt(1.*k);

#ifdef MFTIMINGS
    MFTimeMFCreateHyperCubeAtOrigin+=clock()-starttime;
#endif
  return(result);
 }

MFPolytope MFCreateSimplexAtOrigin(int k, double R)
 {
  static char RoutineName[]={"MFCreateSimplexAtOrigin"};
  int i,j,l;
  MFPolytope result;
  int verbose;
  int l0,m;
  double d;
  double h;
  double Delta;
  double Rd;
  double Rdm;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFCreateSimplexAtOrigin++;
  starttime=clock();
#endif

  verbose=0;
  if(verbose)printf("%s\n",RoutineName);fflush(stdout);

/* Allocate storage */

  result=malloc(sizeof(struct MFPolytopeSt)); /*done*/
  if(result==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFPolytopeSt));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateSimplexAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }
  if(verbose)printf("MFCreateSimplexAtOrigin 0x%8.8x\n",result);

  result->k=k;
  
  result->n=k+1;
  result->m=result->n;
  result->v=malloc(result->n*k*sizeof(double)); /*done*/
  if(result->v==(double*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*k*sizeof(double));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateSimplexAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }
  for(i=0;i<result->n*k;i++)result->v[i]=0.;

  result->v[0+k*0]=-R;for(j=1;j<k;j++)result->v[j+k*0]=0.;
  result->v[0+k*1]= R;for(j=1;j<k;j++)result->v[j+k*1]=0.;
  Rdm=R;

  for(i=2;i<k+1;i++)
   {
    Delta=sqrt(4*R*R-Rdm*Rdm);
    Rd=2*R*R/Delta;

    for(l=0;l<i;l++)result->v[i-1+k*l]=Rd-Delta;
    for(j=0;j<i-1;j++)result->v[j+k*i]=0.;result->v[i-1+k*i]=Rd;for(j=i;j<k;j++)result->v[j+k*i]=0.;

    Rdm=Rd;
   }

/* Scale to contain ball radius R */

  d=0.;h=0.;
  for(j=0;j<k;j++){d+=-result->v[j]*result->v[j+k];h+=result->v[j]*result->v[j];}
  for(i=0;i<k*(k+1);i++)result->v[i]=result->v[i]*R*sqrt(h)/d;

  result->nIndices=malloc(result->n*sizeof(int)); /*done*/
  if(result->nIndices==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateSimplexAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  result->mIndices=malloc(result->n*sizeof(int)); /*done*/
  if(result->mIndices==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateSimplexAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  result->indices=malloc(result->n*sizeof(int*)); /*done*/
  if(result->indices==(int**)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*sizeof(int*));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateSimplexAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }
  if(verbose)printf("   result(0x%8.8x)->indices=0x%8.8x, %d int*s\n",result,result->indices,result->n);

  for(i=0;i<result->n;i++)
   {
    result->nIndices[i]=k;
    result->mIndices[i]=k;
    result->indices[i]=malloc(k*sizeof(int)); /*done*/
    if(result->indices[i]==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",k*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateSimplexAtOrigin+=clock()-starttime;
#endif
      return (MFPolytope)NULL;
     }
    if(verbose)printf("   result(0x%8.8x)->indices[%d]=0x%8.8x, %d ints\n",result,i,result->indices[i],k);
   }

  result->nFaces=k+1;
  result->mFaces=result->nFaces;
  result->face=malloc(result->mFaces*sizeof(int)); /*done*/
  if(result->face==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateSimplexAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  result->nFaceV=malloc(result->mFaces*sizeof(int)); /*done*/
  if(result->nFaceV==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateSimplexAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  result->faceN=malloc(result->mFaces*sizeof(MFKVector)); /*done*/
  if(result->faceN==(MFKVector*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(MFKVector));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateSimplexAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  result->faceO=malloc(result->mFaces*sizeof(double)); /*done*/
  if(result->faceO==(double*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(double));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateSimplexAtOrigin+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }

  d=0;
  for(j=0;j<k;j++){d+=result->v[j]*result->v[j];}
  d=1./sqrt(d);
  for(i=0;i<k+1;i++)
   {
    result->face[i]=i;
    result->nFaceV[i]=0;
    result->faceN[i]=MFCreateKVector(k);
    result->faceO[i]=R;
    for(j=0;j<k;j++)
     {
      MFKVSetC(result->faceN[i],j,-result->v[j+k*i]*d);
      result->nFaceV[j]=k;
     }
    m=0;
    for(j=0;j<k+1;j++)
     {
      if(j!=i)
       {
        (result->indices[i])[m]=j;
        m++;
       }
     }
   }

  result->R=R;

#ifdef MFTIMINGS
    MFTimeMFCreateSimplexAtOrigin+=clock()-starttime;
#endif
  return(result);
 }

void MFFreePolytope(MFPolytope P)
 {
  static char RoutineName[]={"MFFreePolytope"};
  int i;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFFreePolytope++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Polytope (argument 1) is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(4,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFFreePolytope+=clock()-starttime;
#endif
    return;
   }

  if(P->v!=(double *)NULL)free(P->v);
  if(P->nIndices!=(int*)NULL)free(P->nIndices);
  if(P->nIndices!=(int*)NULL)free(P->mIndices);
  for(i=0;i<P->n;i++)
   {
    if(P->indices[i]!=(int*)NULL)free(P->indices[i]);
   }
  if(P->indices!=(int**)NULL)free(P->indices);
  if(P->face!=(int*)NULL)free(P->face);
  if(P->nFaceV!=(int*)NULL)free(P->nFaceV);
  if(P->faceN!=(MFKVector*)NULL)
   {
    for(i=0;i<P->nFaces;i++)MFFreeKVector(P->faceN[i]);
    free(P->faceN);
   }
  if(P->faceO!=(double*)NULL)free(P->faceO);
  free(P);

#ifdef MFTIMINGS
    MFTimeMFFreePolytope+=clock()-starttime;
#endif
  return;
 }

void MFSubtractHalfSpaceFromPolytope(MFPolytope P,int index, MFKVector nrm, double on)
 {
  static char RoutineName[]={"MFSubtractHalfSpaceFromPolytope"};
  int i,j,k,l;
  int n;
  int flag;
  int verbose;
  double *d;
  int *inter;
  int n1,n2;
  double *vi,*vj,*v;
  double t;
  int nold,m,new;
  int ln,li,lj;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFSubtractHalfSpaceFromPolytope++;
  starttime=clock();
#endif

  verbose=0;

  if(verbose)
   {
    printf("MFSubtractHalfSpaceFromPolytope 0x%8.8x\n",P);
    printf("  Polytope\n");
    MFPrintPolytope(stdout,P);
    printf("  HalfSpace\n");

    fprintf(stdout,"                     (%d) x.",index);
    fprintf(stdout,"(");
    for(j=0;j<P->k;j++)
     {
      if(j>0)fprintf(stdout,",");
      t=MFKV_C(nrm,j);
      fprintf(stdout,"%le",t);
     }
    fprintf(stdout,")-%le",on);
    fprintf(stdout,"<=0\n");
   }

  if(!MFPolytopeTestVertexList(P))
   {
    printf("%s line %d, file %s\n",RoutineName,__LINE__,__FILE__);
    fflush(stdout);
    abort();
   }

  if(P->n==0)
   {
#ifdef MFTIMINGS
    MFTimeMFSubtractHalfSpaceFromPolytope+=clock()-starttime;
#endif
    return;
   }

  for(i=0;i<P->nFaces;i++)
   if(P->face[i]==index)
    {
     sprintf(MFPolytopeErrorMsg,"Subtracting a halfspace index already present",index);
     printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
     MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFSubtractHalfSpaceFromPolytope+=clock()-starttime;
#endif
     return;
    }

  d=malloc(P->n*sizeof(double)); /*done*/
  if(d==(double*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->n*sizeof(double));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFSubtractHalfSpaceFromPolytope+=clock()-starttime;
#endif
    return;
   }

  t=fabs(on);
  for(j=0;j<P->k;j++)if(fabs(MFKV_C(nrm,j))>t)t=fabs(MFKV_C(nrm,j));
  if(verbose)printf(" inf norm of (on,norm) is %le\n",t);
  for(i=0;i<P->n;i++)
   {
    d[i]=-on;
    for(j=0;j<P->k;j++)d[i]+=P->v[j+P->k*i]*MFKV_C(nrm,j);
    d[i]=d[i]/t;

    if(verbose)printf("vertex %d is on side %le\n",i,d[i]);
   }

  nold=P->n;
  for(i=0;i<nold;i++)
   {
    n1=P->nIndices[i];
    for(j=i+1;j<nold;j++)
     {

/* This number should be relative to the longest edge of the polytope */

      if(fabs(d[i])>1.e-14 && fabs(d[i])>1.e-14 &&
            (d[i]<0&&d[j]>0 || d[i]>0 && d[j]<0))
       {
        if(verbose)
         {
          printf("  this pair (%d,%d) straddles the new plane\n",i,j);
          printf("    %d indices (",i);
          for(l=0;l<P->nIndices[i];l++){if(l>0)printf(",");printf("%d",(P->indices[i])[l]);}
          printf(") d=%le\n",d[i]);
          printf("    %d indices (",j);
          for(l=0;l<P->nIndices[j];l++){if(l>0)printf(",");printf("%d",(P->indices[j])[l]);}
          printf(") d=%le\n",d[j]);
          t=0.;
          for(l=0;l<P->k;l++)t+=(P->v[l+P->k*i]-P->v[l+P->k*j])*(P->v[l+P->k*i]-P->v[l+P->k*j]);
          t=sqrt(t);
          printf("    edge is length %le\n",t);fflush(stdout);
         }
        n2=P->nIndices[j];
        m=n1;if(n2>m)m=n2;

        if(m>0)
         {
          inter=malloc(m*sizeof(int)); /*done*/
          if(inter==(int*)NULL)
           {
            sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",m*sizeof(int));
            printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
            MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFSubtractHalfSpaceFromPolytope+=clock()-starttime;
#endif
            return;
           }

          n=MFPolytopeIntersectIndexSets(P,i,j,inter);
  
          if(verbose)
           {
            printf("    intersection indices (");
            for(l=0;l<n;l++){if(l>0)printf(",");printf("%d",inter[l]);}
            printf(")\n");
           }
         }else{
          inter=(int*)NULL;
          n=0;
         }

        if(n<(P->k)-1)
         {
          if(verbose)printf("  Not on an edge\n");
          if(inter!=(int*)NULL)free(inter);
          inter=(int*)NULL;
         }else{
          if(verbose)printf("  On an edge\n");
          new=MFPolytopeAddVertex(P);
          t=-d[i]/(d[j]-d[i]);
          li=P->k*i;
          lj=P->k*j;
          ln=P->k*new;
          if(verbose)printf("  d[%d]=%le, d[%d]=%le, t=%le\n",i,d[i],j,d[j],t);
          for(l=0;l<P->k;l++)
           {
            P->v[l+ln]=P->v[l+li]+t*(P->v[l+lj]-P->v[l+li]);
/*          if(verbose)printf("  v[%d]_%d=(vi[%d]_%d)-t*(vj[%d]_%d-vi[%d]_%d)=%le-(%le)*((%le)-(%le))=%le=%le\n",new,l,i,l,j,l,i,l,P->v[l+li],t,P->v[l+lj]-P->v[l+li],P->v[l+li]-t*(P->v[l+lj]-P->v[l+li]),P->v[l+ln]);*/
           }
          P->nIndices[new]=n;
          P->mIndices[new]=m;
          P->indices[new]=inter;
          d=realloc((void*)d,(P->m)*sizeof(double)); /*done*/
          if(d==(double*)NULL)
           {
            sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->m*sizeof(double));
            printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
            MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFSubtractHalfSpaceFromPolytope+=clock()-starttime;
#endif
            return;
           }

          d[new]=-100.;
          MFPolytopeAddVertexIndexIntoSet(P,new,index);
          if(verbose)
           {
            printf("> adding new vertex P(%0x8.8x)->indices[%d]=[",P,new);
            for(l=0;l<P->nIndices[new];l++)
             {
              if(l>0)printf(",");
              printf("%d",(P->indices[new])[l]);
             }
            printf("]\n");
           }
         }
       }
     }
   }

  for(i=0;i<P->n;i++)
   {
    if(d[i]>0)
     {
      if(verbose){printf("Vertex %d is on the wrong side, d[%d]=%le, removing it.\n",i,i,d[i]);fflush(stdout);}
      if(i<P->n-1)
       {
        for(j=0;j<P->k;j++)P->v[j+P->k*i]=P->v[j+P->k*(P->n-1)];
        if(verbose)
         {
          printf("set P->indices[%d]=P->indices[%d]=[",i,P->n-1);
          for(l=0;l<P->nIndices[P->n-1];l++)
           {
            if(l>0)printf(",");
            printf("%d",(P->indices[P->n-1])[l]);
           }
          printf("] d=%le\n",d[P->n-1]);
         }
        if(P->indices[i]!=(int*)NULL)free(P->indices[i]);
        P->indices[i]=P->indices[P->n-1];
        P->nIndices[i]=P->nIndices[P->n-1];
        P->mIndices[i]=P->mIndices[P->n-1];
        d[i]=d[P->n-1];
       }else{
        if(P->indices[i]!=(int*)NULL)free(P->indices[i]);
       }
      P->indices[P->n-1]=(int*)NULL;
      P->nIndices[P->n-1]=0;
      P->mIndices[P->n-1]=0;
      d[P->n-1]=0.;
      (P->n)--;
      i--;
     }else if(d[i]==0)
     {
      if(verbose){printf("Vertex %d is on the hyperplane, adding index to it.\n",i);fflush(stdout);}
      MFPolytopeAddVertexIndexIntoSet(P,i,index);
     }else{
      if(verbose)
       {
        printf("Vertex %d is on the right side, d[%d]=%le, keeping it.\n",i,i,d[i]);
        printf("    P->indices[%d]=[",i);
        for(l=0;l<P->nIndices[i];l++)
         {
          if(l>0)printf(",");
          printf("%d",(P->indices[i])[l]);
         }
        printf("] d=%le\n",d[i]);
       }
     }
   }

  if(d==NULL)printf("freeing(0x0) line %d of file %s\n",__LINE__,__FILE__);
  free(d);

/* Now remove redundant half spaces */
/*   For each appearing index, count occurances. */
/*    If it appears fewer than k-1 times, remove it */
/*    else If dim(span(vertices))<k-1 times, remove it */

  if(P->nFaces>=P->mFaces)
   {
    P->mFaces+=MFPolytopeIncrementToAllocateToFaceList;
    P->face=realloc((void*)P->face,P->mFaces*sizeof(int)); /*done*/
    if(P->face==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->mFaces*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFSubtractHalfSpaceFromPolytope+=clock()-starttime;
#endif
      return;
     }

    P->nFaceV=realloc((void*)P->nFaceV,P->mFaces*sizeof(int)); /*done*/
    if(P->nFaceV==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->mFaces*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFSubtractHalfSpaceFromPolytope+=clock()-starttime;
#endif
      return;
     }

    P->faceN=realloc((void*)(P->faceN),P->mFaces*sizeof(MFKVector)); /*done*/
    if(P->faceN==(MFKVector*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->mFaces*sizeof(MFKVector));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFSubtractHalfSpaceFromPolytope+=clock()-starttime;
#endif
      return;
     }
  
    P->faceO=realloc((void*)P->faceO,P->mFaces*sizeof(double)); /*done*/
    if(P->faceO==(double*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->mFaces*sizeof(double));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFSubtractHalfSpaceFromPolytope+=clock()-starttime;
#endif
      return;
     }
   }

  P->face[P->nFaces]=index;
  P->faceN[P->nFaces]=nrm;MFRefKVector(nrm);
  P->faceO[P->nFaces]=on;
  P->nFaceV[P->nFaces]=0;
  P->nFaces++;
  if(verbose)MFPrintPolytope(stdout,P);
  if(verbose)printf("\n   -- call UpdateFaceList, P has %d vertices \n",P->n);
/*MFPolytopeMergeCloseVertices(P,.01*P->R);*/
  if(!MFPolytopeTestVertexList(P))
   {
    printf("%s line %d, file %s\n",RoutineName,__LINE__,__FILE__);
    fflush(stdout);
    abort();
   }
  MFPolytopeUpdateFaceList(P);

/*if(verbose)MFPrintPolytope(stdout,P);*/
  if(verbose)printf("   done MFSubtractHalfSpaceFromPolytope\n");

#ifdef MFTIMINGS
    MFTimeMFSubtractHalfSpaceFromPolytope+=clock()-starttime;
#endif
  return;
 }

/*----------------------------------------------------------------------*/

int MFPolytopeDimension(MFPolytope P)
 {
  static char RoutineName[]={"MFPolytopeDimension"};

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeDimension++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeDimension+=clock()-starttime;
#endif
    return -1;
   }
#ifdef MFTIMINGS
    MFTimeMFPolytopeDimension+=clock()-starttime;
#endif
  return(P->k);
 }

int MFPolytopeNumberOfVertices(MFPolytope P)
 {
  static char RoutineName[]={"MFPolytopeNumberOfVertices"};

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeNumberOfVertices++;
  starttime=clock();
#endif
  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeNumberOfVertices+=clock()-starttime;
#endif
    return -1;
   }
#ifdef MFTIMINGS
    MFTimeMFPolytopeNumberOfVertices+=clock()-starttime;
#endif
  return(P->n);
 }

void MFPolytopeVertex(MFPolytope P,int i,MFKVector v)
 {
  static char RoutineName[]={"MFPolytopeVertex"};
  int j;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeVertex++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertex+=clock()-starttime;
#endif
    return;
   }

  if(i<0 || i>P->n)
   {
    sprintf(MFPolytopeErrorMsg,"Invalid vertex number %d (must be in [0,%d) )",i,P->n);
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertex+=clock()-starttime;
#endif
    return;
   }

  for(j=0;j<P->k;j++)
    MFKVSetC(v,j,P->v[j+i*P->k]);

#ifdef MFTIMINGS
    MFTimeMFPolytopeVertex+=clock()-starttime;
#endif
  return;
 }

int MFPolytopeNumberOfVertexIndices(MFPolytope P,int i)
 {
  static char RoutineName[]={"MFPolytopeNumberOfVertexIndices"};

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeNumberOfVertexIndices++;
  starttime=clock();
#endif
  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertex+=clock()-starttime;
#endif
    return -1;
   }
  if(i<0 || i>=P->n)
   {
    sprintf(MFPolytopeErrorMsg,"Invalid vertex number %d (must be in [0,%d) )",i,P->n);
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertex+=clock()-starttime;
#endif
    return -1;
   }
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertex+=clock()-starttime;
#endif
  return(P->nIndices[i]);
 }

int MFPolytopeVertexIndex(MFPolytope P,int i,int j)
 {
  static char RoutineName[]={"MFPolytopeVertexIndex"};

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeVertexIndex++;
  starttime=clock();
#endif
  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexIndex+=clock()-starttime;
#endif
    return -1;
   }
  if(i<0 || i>=P->n)
   {
    sprintf(MFPolytopeErrorMsg,"Invalid vertex number %d (must be in [0,%d) )",i,P->n);
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexIndex+=clock()-starttime;
#endif
    return -1;
   }
  if(P->indices==(int**)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Polytope has NULL vertex index list");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexIndex+=clock()-starttime;
#endif
    return -1;
   }
  if(j<0 || j>=P->nIndices[i])
   {
    sprintf(MFPolytopeErrorMsg,"Invalid index number %d (must be in [0,%d) )",j,P->nIndices[i]);
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexIndex+=clock()-starttime;
#endif
    return -1;
   }
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexIndex+=clock()-starttime;
#endif
  return (P->indices[i])[j];
 }

int *MFPolytopeVertexIndexSet(MFPolytope P,int i)
 {
  static char RoutineName[]={"MFPolytopeVertexIndexSet"};
  int j;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeVertexIndexSet++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexIndexSet+=clock()-starttime;
#endif
    return (int*)NULL;
   }
  if(i<0 || i>=P->n)
   {
    sprintf(MFPolytopeErrorMsg,"Invalid vertex number %d (must be in [0,%d) )",i,P->n);
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexIndexSet+=clock()-starttime;
#endif
    return (int*)NULL;
   }

  if(P->indices==(int**)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Polytope has NULL vertex index list");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexIndexSet+=clock()-starttime;
#endif
    return (int*)NULL;
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexIndexSet+=clock()-starttime;
#endif
  return( P->indices[i] );
 }

int MFPolytopeVertexNumberOfIndices(MFPolytope P,int i)
 {
  static char RoutineName[]={"MFPolytopeVertexNumberOfIndices"};
  int j;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeVertexNumberOfIndices++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexNumberOfIndices+=clock()-starttime;
#endif
    return -1;
   }
  if(i<0 || i>=P->n)
   {
    sprintf(MFPolytopeErrorMsg,"Invalid vertex number %d (must be in [0,%d) )",i,P->n);
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexNumberOfIndices+=clock()-starttime;
#endif
    return -1;
   }

  if(P->nIndices==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Polytope has NULL vertex index length list\n");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexNumberOfIndices+=clock()-starttime;
#endif
    return -1;
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexNumberOfIndices+=clock()-starttime;
#endif
  return( P->nIndices[i] );
 }

/*----------------------------------------------------------------------*/

int MFPolytopeAddVertex(MFPolytope P)
 {
  static char RoutineName[]={"MFPolytopeAddVertex"};
  int i;
  int verbose;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeAddVertex++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeAddVertex+=clock()-starttime;
#endif
    return -1;
   }

  verbose=0;

  if(verbose)printf("MFPolytopeAddVertex 0x%8.8x\n",P);

  if(P->n>=P->m)
   {
    if(verbose)printf(" P->n(%d)>=P->m(%d), creating more space\n",P->n,P->m);

    P->m+=MFPolytopeIncrementToAllocateToVertexList;

    P->v=realloc((void*)P->v,P->k*P->m*sizeof(double)); /*done*/
    if(P->v==(double*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->m*sizeof(double));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeAddVertex+=clock()-starttime;
#endif
      return -1;
     }
    for(i=P->k*P->n;i<P->k*P->m;i++)P->v[i]=0.;

    P->nIndices=realloc((void*)P->nIndices,P->m*sizeof(int)); /*done*/
    if(P->nIndices==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->m*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeAddVertex+=clock()-starttime;
#endif
      return -1;
     }
    for(i=P->n;i<P->m;i++)P->nIndices[i]=0;

    P->mIndices=realloc((void*)P->mIndices,P->m*sizeof(int)); /*done*/
    if(P->mIndices==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->m*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeAddVertex+=clock()-starttime;
#endif
      return -1;
     }
    for(i=P->n;i<P->m;i++)P->mIndices[i]=0;

    P->indices=realloc((void*)P->indices,P->m*sizeof(int*)); /*done*/
    if(P->indices==(int**)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->m*sizeof(int*));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeAddVertex+=clock()-starttime;
#endif
      return -1;
     }
    for(i=P->n;i<P->m;i++)P->indices[i]=(int*)NULL;

    if(verbose)printf(" now space for P->m(%d) vertices.\n");
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeAddVertex+=clock()-starttime;
#endif
  if(verbose)printf(" returning vertex %d.\n",P->n);
  (P->n)++;
#ifdef MFTIMINGS
    MFTimeMFPolytopeAddVertex+=clock()-starttime;
#endif
  return(P->n-1);
 }

int MFPolytopeAddVertexIndexIntoSet(MFPolytope P,int v,int index)
 {
  static char RoutineName[]={"MFPolytopeAddVertexIndexIntoSet"};
  int i,j;
  int verbose;
  int *tmp;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeAddVertexIndexIntoSet++;
  starttime=clock();
#endif

  verbose=0;

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeAddVertexIndexIntoSet+=clock()-starttime;
#endif
    return;
   }

  if(v<0 || v>=P->n)
   {
    fprintf(stdout,"ERROR: MFPolytopeAddVertexIndexIntoSet - Invalid face number %d (must be in [0,%d) )\n",v,P->n);
    fprintf(stderr,"MFPolytopeAddVertexIndexIntoSet - Invalid face number %d (must be in [0,%d) )\n",v,P->n);
#ifdef MFTIMINGS
    MFTimeMFPolytopeAddVertexIndexIntoSet+=clock()-starttime;
#endif
    return;
   }

  if(P->nIndices[v]>=P->mIndices[v])
   {
    P->mIndices[v]+=MFPolytopeIncrementToAllocateToIndexSet;
    printf("Realloc P(0x%8.8x)->indices[%d]=0x%8.8x\n",P,v,P->indices[v]);
    P->indices[v]=realloc((void*)P->indices[v],P->mIndices[v]*sizeof(int)); /*done*/
    if(P->indices[v]==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->mIndices[v]*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeAddVertexIndexIntoSet+=clock()-starttime;
#endif
      return -1;
     }
    for(i=P->nIndices[v];i<P->mIndices[v];i++)(P->indices[v])[i]=-1;
   }

/* Insert [9] into [11,49] is wrong? */

  if(verbose)
   {
    printf("Inserting [%d] into [%d",index,(P->indices[v])[0]);
    for(i=1;i<P->nIndices[v];i++)printf(",%d",(P->indices[v])[i]);
    printf("]\n");fflush(stdout);
   }

  j=0;
  while((P->indices[v])[j]<index && j<P->nIndices[v])j++;

  if(verbose)printf("  first index greater than %d is %d, n=%d\n",index,j,P->nIndices[v]);

  for(i=P->nIndices[v]-1;i>=j;i--)
   {
    if(verbose){printf("(P->indices[v])[%d]=(P->indices[v])[%d]=%d;\n",i+1,i,(P->indices[v])[i]);fflush(stdout);}
    (P->indices[v])[i+1]=(P->indices[v])[i];
   }

  if(verbose)
   {
    printf("  shifted [");
    if(j!=0)printf("%d",(P->indices[v])[0]);
    for(i=1;i<P->nIndices[v]+1;i++){printf(",");if(j!=i)printf("%d",(P->indices[v])[i]);}
    printf("]\n");fflush(stdout);
   }

  (P->indices[v])[j]=index;
  (P->nIndices[v])++;

  if(verbose)
   {
    printf("  final [%d",(P->indices[v])[0]);
    for(i=1;i<P->nIndices[v];i++)printf(",%d",(P->indices[v])[i]);
    printf("]\n");fflush(stdout);
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeAddVertexIndexIntoSet+=clock()-starttime;
#endif
  return(j);
 }

int MFPolytopeIntersectIndexSets(MFPolytope P,int v1,int v2,int *inter)
 {
  static char RoutineName[]={"MFPolytopeIntersectIndexSets"};

/* Returns the number of entries two index sets have in common  */
/*   Assumes that the index sets are sorted in increasing order */

  int n;
  int n1,n2;
  int *inter1;
  int *inter2;
  int i1,i2;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeIntersectIndexSets++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeIntersectIndexSets+=clock()-starttime;
#endif
    return;
   }

  n1=P->nIndices[v1];
  inter1=P->indices[v1];
  n2=P->nIndices[v2];
  inter2=P->indices[v2];

  n=0;
  i1=0;i2=0;
  while(i1<n1&&i2<n2)
   {
    if(inter1[i1]==inter2[i2])
     {
      inter[n]=inter1[i1];
      n++;
      i1++;i2++;
     }else if(inter1[i1]<inter2[i2])
     {
      i1++;
     }else if(inter1[i1]>inter2[i2])
     {
      i2++;
     }
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeIntersectIndexSets+=clock()-starttime;
#endif
  return(n);
 }

int MFPolytopeNumberOfFaces(MFPolytope P)
 {
  static char RoutineName[]={"MFPolytopeNumberOfFaces"};

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeNumberOfFaces++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeNumberOfFaces+=clock()-starttime;
#endif
    return;
   }
#ifdef MFTIMINGS
    MFTimeMFPolytopeNumberOfFaces+=clock()-starttime;
#endif
  return(P->nFaces);
 }

int MFPolytopeFaceIndex(MFPolytope P,int face)
 {
  static char RoutineName[]={"MFPolytopeFaceIndex"};

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeFaceIndex++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeFaceIndex+=clock()-starttime;
#endif
    return;
   }
#ifdef MFTIMINGS
    MFTimeMFPolytopeFaceIndex+=clock()-starttime;
#endif
  return(P->face[face]);
 }

MFKVector MFPolytopeFaceNormal(MFPolytope P,int face)
 {
  static char RoutineName[]={"MFPolytopeFaceNormal"};

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeFaceNormal++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeFaceNormal+=clock()-starttime;
#endif
    return;
   }
#ifdef MFTIMINGS
    MFTimeMFPolytopeFaceNormal+=clock()-starttime;
#endif
  return(P->faceN[face]);
 }

double MFPolytopeFaceOrigin(MFPolytope P,int face)
 {
  static char RoutineName[]={"MFPolytopeFaceOrigin"};

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeFaceOrigin++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeFaceOrigin+=clock()-starttime;
#endif
    return;
   }
#ifdef MFTIMINGS
    MFTimeMFPolytopeFaceOrigin+=clock()-starttime;
#endif
  return(P->faceO[face]);
 }

int MFPolytopeNumberOfVerticesOnFace(MFPolytope P,int face)
 {
  static char RoutineName[]={"MFPolytopeNumberOfVerticesOnFace"};

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeNumberOfVerticesOnFace++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeNumberOfVerticesOnFace+=clock()-starttime;
#endif
    return;
   }
#ifdef MFTIMINGS
    MFTimeMFPolytopeNumberOfVerticesOnFace+=clock()-starttime;
#endif
  return(P->nFaceV[face]);
 }

double MFPolytopeRadiusOfVertex(MFPolytope P,int vertex)
 {
  static char RoutineName[]={"MFPolytopeRadiusOfVertex"};
  int i;
  double r;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeRadiusOfVertex++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeRadiusOfVertex+=clock()-starttime;
#endif
    return;
   }

  if(vertex<0 || vertex>=P->n)
   {
    sprintf(MFPolytopeErrorMsg,"Invalid vertex=%d, P->n=%d\n",vertex,P->n);
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeRadiusOfVertex+=clock()-starttime;
#endif
    return;
   }

  r=0.;
  for(i=0;i<P->k;i++)
   {
    r+=P->v[i+P->k*vertex]*P->v[i+P->k*vertex];
   }
  r=sqrt(r);
#ifdef MFTIMINGS
    MFTimeMFPolytopeRadiusOfVertex+=clock()-starttime;
#endif
  return(r);
 }

void MFPolytopeUpdateFaceList(MFPolytope P)
 {
  static char RoutineName[]={"MFPolytopeUpdateFaceList"};
  int i,j;
  int index;
  int l,jj;
  int verbose;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeUpdateFaceList++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeUpdateFaceList+=clock()-starttime;
#endif
    return;
   }

  verbose=0;
  if(verbose)printf("MFPolytopeUpdateFaceList 0x%8.8x\n",P);

  if(verbose)printf(" This polytope has %d vertices and %d faces\n",P->n,P->nFaces);
  for(i=0;i<P->nFaces;i++)P->nFaceV[i]=0;

  for(i=0;i<P->n;i++)
   {
    if(verbose){printf(" Vertex %d has %d indices\n",i,P->nIndices[i]);fflush(stdout);}
    if(verbose){printf("\nP->nIndices[%d]=%d, P->indices[%d]=0x%8.8x\n",i,P->nIndices[i],i,P->indices[i]);fflush(stdout);}
    for(j=0;j<P->nIndices[i];j++)
     {
      if(verbose){printf("   Index %d is %d\n",j,(P->indices[i])[j]);fflush(stdout);}
      index=(P->indices[i])[j];
      jj=-1;
      for(l=0;l<P->nFaces;l++)if(P->face[l]==index)jj=l;
      if(jj>-1)
       {
        if(verbose){printf("   face %d has this index\n",jj);fflush(stdout);}
        P->nFaceV[jj]++;
       }
     }
   }

#ifdef MFREMOVEREDUNDANTFACES
  if(verbose){printf("Now remove faces with too few points\n");fflush(stdout);}
  for(i=0;i<P->nFaces;i++)
   {
    if(verbose){printf(" Face %d contains %d vertices\n",i,P->nFaceV[i]);fflush(stdout);}
    if(verbose)printf("This Polytope now has %d faces\n",P->nFaces);
    if(P->nFaceV[i]<P->k)
     {
      if(verbose){printf(" Removing face %d\n",P->face[i]);fflush(stdout);}
      for(j=0;j<P->n;j++)
       {
        if(verbose)
         {
          printf(" Vertex %d has %d indices [",j,P->nIndices[j]);
          for(jj=0;jj<P->nIndices[j];jj++)
           {
            if(jj>0)printf(",");
            printf("%d",(P->indices[j])[jj]);
           }
          printf("]\n");
          fflush(stdout);
         }
        for(jj=0;jj<P->nIndices[j];jj++)
         {
          if((P->indices[j])[jj]==P->face[i])
           {
            if(verbose){printf("   index %d is the face being removed, moving index set down\n",jj);fflush(stdout);}
            for(l=jj;l<P->nIndices[j]-1;l++)
             {
              (P->indices[j])[l]=(P->indices[j])[l+1];
             }
            P->nIndices[j]--;
            jj--;
           }
         }
        if(verbose)
         {
          printf(" >Vertex %d now has %d indices [",j,P->nIndices[j]);
          for(jj=0;jj<P->nIndices[j];jj++)
           {
            if(jj>0)printf(",");
            printf("%d",(P->indices[j])[jj]);
           }
          printf("]\n");
          fflush(stdout);
         }
       }
      P->nFaces--;
      P->face[i]=P->face[P->nFaces];
      P->nFaceV[i]=P->nFaceV[P->nFaces];
      if(P->faceN[i]!=(MFKVector)NULL)MFFreeKVector(P->faceN[i]);
      P->faceN[i]=P->faceN[P->nFaces];
      P->faceO[i]=P->faceO[P->nFaces];
      i--;
     }
   }
#endif
  if(verbose)printf("done MFPolytopeUpdateFaceList\n");

#ifdef MFTIMINGS
    MFTimeMFPolytopeUpdateFaceList+=clock()-starttime;
#endif
  return;
 }

int MFPolytopeVerticesOnSameEdge(MFPolytope P,int i,int j)
 {
  static char RoutineName[]={"MFPolytopeVerticesOnSameEdge"};
  int n1,n2;
/*int *inter1,*inter2;*/
  int *inter;
  int n,m,result;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeVerticesOnSameEdge++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVerticesOnSameEdge+=clock()-starttime;
#endif
    return;
   }

  n1=P->nIndices[i];
  n2=P->nIndices[j];
/*inter1=P->indices[i];
  inter2=P->indices[j];*/
  m=n1;if(n2>m)m=n2;
  inter=malloc(m*sizeof(int)); /*done*/
  if(inter==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",m*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVerticesOnSameEdge+=clock()-starttime;
#endif
    return -1;
   }
/*n=MFPolytopeIntersectIndexSets(n1,inter1,n2,inter2,inter);*/
  n=MFPolytopeIntersectIndexSets(P,i,j,inter);

  printf("   free inter =0x%8.8x\n",inter);fflush(stdout);
  if(inter!=NULL)free(inter);
  free(inter);
  result=n>=P->k-1;
#ifdef MFTIMINGS
    MFTimeMFPolytopeVerticesOnSameEdge+=clock()-starttime;
#endif
  return(result);
 }

int MFPolytopeVerticesOnEdgeWithIndexLessThan(MFPolytope P,int i,int j,int e)
 {
  static char RoutineName[]={"MFPolytopeVerticesOnEdgeWithIndexLessThan"};
  int n1,n2;
/*int *inter1,*inter2;*/
  int *inter;
  int n,m,result;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeVerticesOnEdgeWithIndexLessThan++;
  starttime=clock();
#endif

  n1=P->nIndices[i];
  n2=P->nIndices[j];
/*inter1=P->indices[i];
  inter2=P->indices[j];*/
  m=n1;if(n2>m)m=n2;
  inter=malloc(m*sizeof(int)); /*done*/
  if(inter==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",m*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVerticesOnEdgeWithIndexLessThan+=clock()-starttime;
#endif
    return -1;
   }
/*n=MFPolytopeIntersectIndexSets(n1,inter1,n2,inter2,inter);*/
  n=MFPolytopeIntersectIndexSets(P,i,j,inter);

  result=1;
  for(m=0;m<n;m++)if(inter[m]>=e)result=0;

  printf("   free inter =0x%8.8x\n",inter);fflush(stdout);
  if(inter!=NULL)free(inter);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVerticesOnEdgeWithIndexLessThan+=clock()-starttime;
#endif
  return(result);
 }

int MFPolytopeVertexLargestIndex(MFPolytope P,int i)
 {
  static char RoutineName[]={"MFPolytopeVertexLargestIndex"};
  int j;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeVertexLargestIndex++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexLargestIndex+=clock()-starttime;
#endif
    return;
   }

  if(i<0 || i>=P->n)
   {
    sprintf(MFPolytopeErrorMsg,"Invalid vertex number %d (must be in [0,%d) )",i,P->n);
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexLargestIndex+=clock()-starttime;
#endif
    return;
   }

  if(P->indices==(int**)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Polytope has NULL vertex index list");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexLargestIndex+=clock()-starttime;
#endif
    return;
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexLargestIndex+=clock()-starttime;
#endif
  return(( P->indices[i])[P->nIndices[i]-1] );
 }

int MFPolytopeVertexSmallestIndex(MFPolytope P,int i)
 {
  static char RoutineName[]={"MFPolytopeVertexSmallestIndex"};
  int j;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeVertexSmallestIndex++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexSmallestIndex+=clock()-starttime;
#endif
    return;
   }

  if(i<0 || i>=P->n)
   {
    sprintf(MFPolytopeErrorMsg,"Invalid vertex number %d (must be in [0,%d) )",i,P->n);
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexSmallestIndex+=clock()-starttime;
#endif
    return;
   }

  if(P->indices==(int**)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Polytope has NULL vertex index list");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexSmallestIndex+=clock()-starttime;
#endif
    return;
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeVertexSmallestIndex+=clock()-starttime;
#endif
  return(( P->indices[i])[0] );
 }

int MFPolytopeLargestVertexIndex(MFPolytope P)
 {
  static char RoutineName[]={"MFPolytopeLargestVertexIndex"};
  int i,n;
  int t;
  int result;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeLargestVertexIndex++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeLargestVertexIndex+=clock()-starttime;
#endif
    return;
   }

  n=MFPolytopeNumberOfVertices(P);
  if(n>0)result=MFPolytopeVertexLargestIndex(P,0);
   else  result=-1;

  for(i=1;i<n;i++)
   {
    t=MFPolytopeVertexLargestIndex(P,i);
    if(t>result)result=t;
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeLargestVertexIndex+=clock()-starttime;
#endif
  return(result);
 }

int MFPolytopeSmallestVertexIndex(MFPolytope P)
 {
  static char RoutineName[]={"MFPolytopeSmallestVertexIndex"};
  int i,n;
  int t;
  int result;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeSmallestVertexIndex++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeLargestVertexIndex+=clock()-starttime;
#endif
    return;
   }

  n=MFPolytopeNumberOfVertices(P);
  if(n>0)result=MFPolytopeVertexSmallestIndex(P,0);
   else  result=-1;

  for(i=1;i<n;i++)
   {
    t=MFPolytopeVertexSmallestIndex(P,i);
    if(t<result)result=t;
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeLargestVertexIndex+=clock()-starttime;
#endif
  return(result);
 }

int MFPolytopeInterior(MFPolytope P, MFKVector s)
 {
  static char RoutineName[]={"MFPolytopeInterior"};
  int face;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeInterior++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
    return -1;
   }

  if(s==(MFKVector)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Second argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
    return -1;
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
  if(P->n==0)return 0;
  for(face=0;face<P->nFaces;face++)
   {
    if(MFKVDot(s,P->faceN[face])-P->faceO[face]>0.)
     {
#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
      return 0;
     }
   }
#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
  return 1;
 }

int MFPolytopeTotallyInterior(MFPolytope P, MFKVector s, double eps)
 {
  static char RoutineName[]={"MFPolytopeTotallyInterior"};
  int face;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeTotallyInterior++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
    return;
   }

  if(s==(MFKVector)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Second argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
    return;
   }

  if(P->n==0)
   {
#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
    return 0;
   }
  for(face=0;face<P->nFaces;face++)
   {
     if(MFKVDot(s,P->faceN[face])-P->faceO[face]>-eps)
      {
#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
       return 0;
      }
    }
#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
  return 1;
 }

double MFPolytopeLargestRadiusOfVertex(MFPolytope P)
 {
  static char RoutineName[]={"MFPolytopeLargestRadiusOfVertex"};
  double r,result;
  int i;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeLargestRadiusOfVertex++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
    return;
   }

  result=0;
  for(i=0;i<P->n;i++)
   if((r=MFPolytopeRadiusOfVertex(P,i))>result)result=r;

#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
  return result;
 }

void MFWritePolytope(FILE *fid,MFPolytope P)
 {
  static char RoutineName[]={"MFWritePolytope"};
  int i,j;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFWritePolytope++;
  starttime=clock();
#endif

  fprintf(fid,"%s\n","Polytope");
  if(P==(MFPolytope)NULL)
   {
    fprintf(fid,"-1\n");
#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
    return;
   }
  fprintf(fid,"%d\n",P->k);

  fprintf(fid,"%d %d\n",P->n,P->m);
  for(i=0;i<P->n*P->k;i++)
   {
    if(i>0)fprintf(fid," ");
    fprintf(fid,"%le",P->v[i]);
   }
  fprintf(fid,"\n");
  for(i=0;i<P->n;i++)
   {
    if(i>0)fprintf(fid," ");
    fprintf(fid,"%d",P->nIndices[i]);
   }
  fprintf(fid,"\n");
  for(i=0;i<P->n;i++)
   {
    if(i>0)fprintf(fid," ");
    fprintf(fid,"%d",P->mIndices[i]);
   }
  fprintf(fid,"\n");
  for(i=0;i<P->n;i++)
   {
    for(j=0;j<P->nIndices[i];j++)
     {
      if(j>0)fprintf(fid," ");
      fprintf(fid,"%d",(P->indices[i])[j]);
     }
    fprintf(fid,"\n");
   }

  fprintf(fid,"%d %d\n",P->nFaces,P->mFaces);
  for(i=0;i<P->nFaces;i++)
   {
    if(i>0)fprintf(fid," ");
    fprintf(fid,"%d",P->face[i]);
   }
  fprintf(fid,"\n");
  for(i=0;i<P->nFaces;i++)
   {
    if(i>0)fprintf(fid," ");
    fprintf(fid,"%d",P->nFaceV[i]);
   }
  fprintf(fid,"\n");
  for(i=0;i<P->nFaces;i++)
   {
    MFWriteKVector(fid,P->faceN[i]);
   }
  fprintf(fid,"\n");
  for(i=0;i<P->nFaces;i++)
   {
    if(i>0)fprintf(fid," ");
    fprintf(fid,"%le",P->faceO[i]);
   }
  fprintf(fid,"\n");

#ifdef MFTIMINGS
    MFTimeMFPolytopeInterior+=clock()-starttime;
#endif
  return;
 }

MFPolytope MFReadPolytope(FILE *fid)
 {
  static char RoutineName[]={"MFReadPolytope"};
  int i,j;
  MFPolytope P;
  char tag[100]="";
  int k=0;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFReadPolytope++;
  starttime=clock();
#endif

  fscanf(fid,"%s\n",tag);
  if(strcmp(tag,"Polytope"))
   {
    sprintf(MFPolytopeErrorMsg,"Next Object is not a Polytope! (%s)\n",RoutineName,tag);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
    return;
   }

  fscanf(fid,"%d\n",&k);
  if(k<0)
   {
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
    return((MFPolytope)NULL);
   }

  P=malloc(sizeof(struct MFPolytopeSt)); /*done*/
  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFPolytopeSt));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
    return (MFPolytope)NULL;
   }
  P->k=k;

  fscanf(fid,"%d %d\n",&(P->n),&(P->m));

  if(P->n>0)
   {
    P->v=malloc(P->n*P->k*sizeof(double)); /*done*/
    if(P->v==(double*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->n*P->k*sizeof(double));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
      return (MFPolytope)NULL;
     }
    for(i=0;i<P->n*P->k;i++)
     {
      if(i>0)fscanf(fid," ");
      fscanf(fid,"%le",&(P->v[i]));
     }
    fscanf(fid,"\n");

    P->nIndices=malloc(P->n*sizeof(int)); /*done*/
    if(P->nIndices==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->n*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
      return (MFPolytope)NULL;
     }
    for(i=0;i<P->n;i++)
     {
      if(i>0)fscanf(fid," ");
      fscanf(fid,"%d",&(P->nIndices[i]));
     }
    fscanf(fid,"\n");

    P->mIndices=malloc(P->n*sizeof(int)); /*done*/
    if(P->mIndices==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->n*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
      return (MFPolytope)NULL;
     }
    for(i=0;i<P->n;i++)
     {
      if(i>0)fscanf(fid," ");
      fscanf(fid,"%d",&(P->mIndices[i]));
     }
    fscanf(fid,"\n");

    P->indices=malloc(P->n*sizeof(int*)); /*done*/
    if(P->indices==(int**)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->n*sizeof(int*));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
      return (MFPolytope)NULL;
     }
  
    for(i=0;i<P->n;i++)
     {
      P->indices[i]=malloc(P->mIndices[i]*sizeof(int)); /*done*/
      if(P->indices[i]==(int*)NULL)
       {
        sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->mIndices[i]*sizeof(int));
        printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
        MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
        return (MFPolytope)NULL;
       }
      for(j=0;j<P->nIndices[i];j++)
       {
        if(j>0)fscanf(fid," ");
        fscanf(fid,"%d",&((P->indices[i])[j]));
       }
      fscanf(fid,"\n");
     }
   }else{
    P->v=(double*)NULL;
    P->nIndices=(int*)NULL;
    P->mIndices=(int*)NULL;
    P->indices=(int**)NULL;
   }

  fscanf(fid,"%d %d\n",&(P->nFaces),&(P->mFaces));

  if(P->nFaces>0)
   {
    P->face=malloc(P->mFaces*sizeof(int)); /*done*/
    if(P->face==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->mFaces*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
      return (MFPolytope)NULL;
     }
    for(i=0;i<P->nFaces;i++)
     {
      if(i>0)fscanf(fid," ");
      fscanf(fid,"%d",&(P->face[i]));
     }
    fscanf(fid,"\n");

    P->nFaceV=malloc(P->mFaces*sizeof(int)); /*done*/
    if(P->nFaceV==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->mFaces*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
      return (MFPolytope)NULL;
     }
    for(i=0;i<P->nFaces;i++)
     {
      if(i>0)fscanf(fid," ");
      fscanf(fid,"%d",&(P->nFaceV[i]));
     }
    fscanf(fid,"\n");

    P->faceN=malloc(P->mFaces*sizeof(MFKVector)); /*done*/
    if(P->faceN==(MFKVector*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->mFaces*sizeof(MFKVector));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
      return;
     }
    for(i=0;i<P->nFaces;i++)
     {
      P->faceN[i]=MFReadKVector(fid);
     }
    fscanf(fid,"\n");

    P->faceO=malloc(P->mFaces*sizeof(double)); /*done*/
    if(P->faceO==(double*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->mFaces*sizeof(double));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
      return (MFPolytope)NULL;
     }
    for(i=0;i<P->nFaces;i++)
     {
      if(i>0)fscanf(fid," ");
      fscanf(fid,"%le",&(P->faceO[i]));
     }
    fscanf(fid,"\n");
   }else{
    P->face=(int*)NULL;
    P->nFaceV=(int*)NULL;
    P->faceN=(MFKVector*)NULL;
    P->faceO=(double*)NULL;
   }

#ifdef MFTIMINGS
    MFTimeMFReadPolytope+=clock()-starttime;
#endif
  return P;
 }

int MFTestPolytope(MFPolytope P)
 {
  static char RoutineName[]={"MFTestPolytope"};
  int coordinate;
  int vertex;
  int face;
  MFKVector s;
  int good;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFTestPolytope++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFTestPolytope+=clock()-starttime;
#endif
    return;
   }

  s=MFCreateKVector(P->k);

  good=1;
  for(vertex=0;vertex<P->n;vertex++)
   {
    for(coordinate=0;coordinate<P->k;coordinate++)MFKVSetC(s,coordinate,P->v[coordinate+vertex*P->k]);
    for(face=0;face<P->nFaces;face++)
     {
      if(MFKVDot(s,P->faceN[face])>P->faceO[face]+1.e-5)
       {
        printf("ERROR, Here's a polytope vertex (%d) that lies completely outside the planes (%d)!\n",vertex,face);
        printf("       ");
        MFPrintKVector(stdout,s);
        printf(".");
        MFPrintKVector(stdout,P->faceN[face]);
        printf("= %le > %le\n",MFKVDot(s,P->faceN[face]),P->faceO[face]);
        good=0;
       }
     }
   }

  if(!good)
   {
    MFPrintPolytope(stdout,P);fflush(stdout);
    abort();
   }
  MFFreeKVector(s);
#ifdef MFTIMINGS
    MFTimeMFTestPolytope+=clock()-starttime;
#endif
  return;
 }

void MFPolytopeMergeCloseVertices(MFPolytope P,double eps)
 {
  static char RoutineName[]={"MFPolytopeMergeCloseVertices"};
  int i,j,l;
  double dist;
  int nElim;
  int verbose;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeMergeCloseVertices++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeMergeCloseVertices+=clock()-starttime;
#endif
    return;
   }

  verbose=0;
  if(verbose)printf("MFPolytopeMergeCloseVertices\n");

  nElim=0;
  loop:
  for(i=0;i<P->n;i++)
   {
    for(j=0;j<P->n;j++)
     {
      if(i!=j)
       {
        dist=0.;
        for(l=0;l<P->k;l++)
         dist+=fabs(P->v[l+i*P->k]-P->v[l+j*P->k]);
        if(dist/P->k<eps)
         {
          nElim++;
          if(verbose)
           {
            printf("MFPolytopeMergeCloseVertices: Merging vertices %d and %d\n",i,j);
            printf("  before:");
            MFPrintPolytope(stdout,P);
           }
          MFPolytopeMergeVertices(P,i,j);
          if(verbose)
           {
            printf("  after:");
            MFPrintPolytope(stdout,P);
           }
          goto loop;
         }
       }
     }
   }
  if(verbose&&nElim>0)printf("MFPolytopeMergeCloseVertices: Merged %d vertices\n",nElim);
#ifdef MFTIMINGS
    MFTimeMFPolytopeMergeCloseVertices+=clock()-starttime;
#endif
  return;
 }

void MFPolytopeMergeVertices(MFPolytope P,int i,int j)
 {
  static char RoutineName[]={"MFPolytopeMergeVertices"};
  int n1,n2,n;
  int l;
  int *Union;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeMergeVertices++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeMergeVertices+=clock()-starttime;
#endif
    return;
   }

/*if(!MFTestVertexIndexOrder(P->nIndices[i],P->indices[i]))
   {
    printf("Error: MFPolytopeMergeVertices at entry vertex indices not sorted\n");
    fflush(stdout);
   }
  if(!MFTestVertexIndexOrder(P->nIndices[j],P->indices[j]))
   {
    printf("Error: MFPolytopeMergeVertices at entry vertex indices not sorted\n");
    fflush(stdout);
   }*/

  n1=P->nIndices[i];
  n2=P->nIndices[j];
  n=n1+n2;
  Union=malloc(n*sizeof(int)); /*done*/
  if(Union==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",n*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeMergeVertices+=clock()-starttime;
#endif
    return;
   }
  n=MFPolytopeUnionOfIndexSets(P,i,j,Union);
  P->nIndices[i]=n;
  printf("   free P(0x%8.8x)->indices[%d]=0x%8.8x\n",P,i,P->indices[i]);fflush(stdout);
  free(P->indices[i]);
  P->indices[i]=Union;
  if(j<P->n-1)
   {
    printf("%s replace P(0x%8.8x)->indices[%d]=0x%8.8x with P(0x%8.8x)->indices[%d]=0x%8.8x\n",RoutineName,P,j,P->indices[j],P,P->n-1,P->indices[P->n-1]);fflush(stdout);
    printf("   free P(0x%8.8x)->indices[%d]=0x%8.8x\n",P,j,P->indices[j]);fflush(stdout);
    free(P->indices[j]);
    P->nIndices[j]=P->nIndices[P->n-1];
    P->indices[j]=P->indices[P->n-1];
    for(l=0;l<P->k;l++)
      P->v[l+P->k*j]=P->v[l+P->k*(P->n-1)];
   }
  (P->n)--;

/*if(!MFTestVertexIndexOrder(P->nIndices[i],P->indices[i]))
   {
    printf("Error: MFPolytopeMergeVertices at exit vertex indices not sorted\n");
    fflush(stdout);
   }*/

#ifdef MFTIMINGS
    MFTimeMFPolytopeMergeVertices+=clock()-starttime;
#endif
  return;
 }

int MFPolytopeUnionOfIndexSets(MFPolytope P,int v1,int v2,int *inter)
 {
  static char RoutineName[]={"MFPolytopeUnionOfIndexSets"};

/* Returns the number of entries two index sets have in common  */
/*   Assumes that the index sets are sorted in increasing order */

  int n;
  int n1,n2;
  int *inter1;
  int *inter2;
  int i1,i2;
  int i;
  int verbose=0;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeUnionOfIndexSets++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeUnionOfIndexSets+=clock()-starttime;
#endif
    return;
   }

  n1=P->nIndices[v1];
  inter1=P->indices[v1];
  n2=P->nIndices[v2];
  inter2=P->indices[v2];
  if(verbose)
   {
    printf("%s\n",RoutineName);
    printf("set 1: [");
    for(i=0;i<n1;i++){if(i>0)printf(",");printf("%d",inter1[i]);}
    printf("]\n");
    printf("set 2: [");
    for(i=0;i<n2;i++){if(i>0)printf(",");printf("%d",inter2[i]);}
    printf("]\n");
   }

  n=0;
  i1=0;i2=0;
  while(i1<n1||i2<n2)
   {
    if(i1>=n1)
     {
      if(verbose)printf("take 2 %d (out of 1)\n",inter2[i2]);
      inter[n]=inter2[i2];
      n++;
      i2++;
    }else if(i2>=n2)
     {
      if(verbose)printf("take 1 %d (out of 2)\n",inter1[i2]);
      inter[n]=inter1[i1];
      n++;
      i1++;
     }else if(inter1[i1]<inter2[i2])
     {
      if(verbose)printf("take 1 %d (smaller than 2 %d)\n",inter1[i1],inter2[i2]);
      inter[n]=inter1[i1];
      n++;
      i1++;
     }else if(inter2[i2]<inter1[i1])
     {
      if(verbose)printf("take 2 %d (smaller than 1 %d)\n",inter2[i1],inter1[i2]);
      inter[n]=inter2[i2];
      n++;
      i2++;
     }else{
      if(verbose)printf("take 2 %d (same as 1 %d)\n",inter2[i1],inter1[i2]);
      inter[n]=inter2[i2];
      n++;
      i1++;
      i2++;
     }
   }

  if(verbose)
   {
    printf("union: [");
    for(i=0;i<n;i++){if(i>0)printf(",");printf("%d",inter[i]);}
    printf("]\n");fflush(stdout);
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeUnionOfIndexSets+=clock()-starttime;
#endif
  return(n);
 }

int MFTestVertexIndexOrder(int n,int *s)
 {
  static char RoutineName[]={"MFTestVertexIndexOrder"};
  int i;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFTestVertexIndexOrder++;
  starttime=clock();
#endif

  for(i=1;i<n;i++)
   if(s[i-1]>=s[i])
    {
#ifdef MFTIMINGS
    MFTimeMFTestVertexIndexOrder+=clock()-starttime;
#endif
     return 0;
    }
#ifdef MFTIMINGS
    MFTimeMFTestVertexIndexOrder+=clock()-starttime;
#endif
  return 1;
 }

int MFPolytopeTestVertexList(MFPolytope P)
 {
  static char RoutineName[]={"MFPolytopeTestVertexList"};
  int i,j,k,n,m;
  int same;
  int good;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeTestVertexList++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFTestVertexIndexOrder+=clock()-starttime;
#endif
    return;
   }
  
  n=P->n;
  good=1;
  for(i=0;i<n-1;i++)
   {
    for(j=i+1;j<n;j++)
     {
      if(P->nIndices[i]==P->nIndices[j])
       {
        m=P->nIndices[i];
        same=1;
        for(k=0;k<m;k++)
         {
          if((P->indices[i])[k]!=(P->indices[j])[k])same=0;
         }
        if(same)
         {
          printf("Problem with Polytope 0x%8.8x, two vertices %d and %d with the same index set (",P,i,j);
          for(k=0;k<m;k++)
           {
            if(k>0)printf(",");
            printf("%d",(P->indices[i])[k]);
           }
          printf("),(");
          for(k=0;k<m;k++)
           {
            if(k>0)printf(",");
            printf("%d",(P->indices[j])[k]);
           }
          printf(")\n");
          fflush(stdout);
          good=0;
         }
       }
     }
   }
  
  for(i=0;i<n;i++)
   {
    m=P->nIndices[i];
    for(j=0;j<m-1;j++)
     {
      for(k=j+1;k<m;k++)
       {
        if((P->indices[i])[j]==(P->indices[i])[k])
         {
          printf("Problem with Polytope 0x%8.8x, vertex %d has a repeated index (",P,i);
          for(k=0;k<m;k++)
           {
            if(k>0)printf(",");
            printf("%d",(P->indices[i])[k]);
           }
          printf(")\n");
         }
       }
     }
   }
  if(!good)
   {
    MFPrintPolytope(stdout,P);fflush(stdout);
   }

#ifdef MFTIMINGS
    MFTimeMFTestVertexIndexOrder+=clock()-starttime;
#endif
  return good;
 }

#ifdef READY
MFPolytope MFCreateSectionOfPolytope(MFPolytope P, MFKVector nrm, double on)
 {
  static char RoutineName[]={"MFCreateSectionOfPolytope"};
  MFPolytope result;
  int i,j;
  int k;
  double *d;
  double t;

  result=malloc(sizeof(struct MFPolytopeSt)); /*done*/
  if(result==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFPolytopeSt));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
    return (MFPolytope)NULL;
   }

  k=P->k;

  result->k=k-1;
  
  result->n=1;for(i=0;i<k;i++)result->n*=2;
  result->m=result->n;
  result->v=malloc(result->n*k*sizeof(double)); /*done*/
  if(result->v==(double*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*k*sizeof(double));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
    return (MFPolytope)NULL;
   }
  for(i=0;i<result->n*k;i++)result->v[i]=0.;

  result->nIndices=malloc(result->n*sizeof(int)); /*done*/
  if(result->nIndices==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
    return (MFPolytope)NULL;
   }

  result->mIndices=malloc(result->n*sizeof(int)); /*done*/
  if(result->mIndices==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
    return (MFPolytope)NULL;
   }

  result->indices=malloc(result->n*sizeof(int*)); /*done*/
  if(result->indices==(int**)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->n*sizeof(int*));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
    return (MFPolytope)NULL;
   }
  if(verbose)printf("   result(0x%8.8x)->indices=0x%8.8x, %d int*s\n",result,result->indices,result->n);
  for(i=0;i<result->n;i++)
   {
    result->nIndices[i]=0;
    result->mIndices[i]=k;
    result->indices[i]=malloc(k*sizeof(int)); /*done*/
    if(result->indices[i]==(int*)NULL)
     {
      sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",k*sizeof(int));
      printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
      MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
      return (MFPolytope)NULL;
     }
    if(verbose)printf("   result(0x%8.8x)->indices[%d]=0x%8.8x, %d ints\n",result,i,result->indices[i],k);
   }

  for(i=0;i<result->n;i++)
   {
    l=i*k;
    if(verbose)printf("vertex %d\n",i);
    for(j=0;j<k;j++)result->v[j+l]=c[j];
    if(verbose){printf(" v=[");for(j=0;j<k;j++){if(j>0)printf(",");printf("%lf",result->v[j+l]);}printf("]\n");}

    result->nIndices[i]=k;
    result->mIndices[i]=k;
    for(j=0;j<k;j++)
     {
      if(c[j]<0)(result->indices[i])[j]=2*j;
       else (result->indices[i])[j]=2*j+1;
     }
    if(verbose){printf(" i=[");for(j=0;j<k;j++){if(j>0)printf(",");printf("%d",(result->indices[i])[j]);}printf("]\n");}

  result->nFaces=2*k;
  result->mFaces=result->nFaces;
  result->face=malloc(result->mFaces*sizeof(int)); /*done*/
  if(result->face==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
    return (MFPolytope)NULL;
   }

  result->nFaceV=malloc(result->mFaces*sizeof(int)); /*done*/
  if(result->nFaceV==(int*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(int));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
    return (MFPolytope)NULL;
   }

  result->faceN=malloc(result->mFaces*sizeof(MFKVector)); /*done*/
  if(result->faceN==(MFKVector*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(MFKVector));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
    return (MFPolytope)NULL;
   }

  result->faceO=malloc(result->mFaces*sizeof(double)); /*done*/
  if(result->faceO==(double*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",result->mFaces*sizeof(double));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
    return (MFPolytope)NULL;
   }

  for(i=0;i<2*k;i++)
   {
    result->face[i]=i;
    result->nFaceV[i]=0;
    result->faceN[i]=MFCreateKVector(k);
    if(i%2==0)
      MFKVSetC(result->faceN[i],i/2,R);
     else
      MFKVSetC(result->faceN[i],i/2,-R);
    result->faceO[i]=1.;
   }

  for(i=0;i<result->n;i++)
   {
    for(j=0;j<result->nIndices[i];j++)
     {
      result->nFaceV[(result->indices[i])[j]]++;
     }
   }

  result->R=P->R;

  d=malloc(P->n*sizeof(double)); /*done*/
  if(d==(double*)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Out of memory, trying to allocate %d bytes",P->n*sizeof(double));
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
    return;
   }

  t=fabs(on);
  for(j=0;j<P->k;j++)if(fabs(MFKV_C(nrm,j))>t)t=fabs(MFKV_C(nrm,j));
  if(verbose)printf(" inf norm of (on,norm) is %le\n",t);
  for(i=0;i<P->n;i++)
   {
    d[i]=-on;
    for(j=0;j<P->k;j++)d[i]+=P->v[j+P->k*i]*MFKV_C(nrm,j);
    d[i]=d[i]/t;

    if(verbose)printf("vertex %d is on side %le\n",i,d[i]);
   }

  return result;
 }
#endif

int MFPolytopeClosestFace(MFPolytope P, MFKVector s, double *d)
 {
  static char RoutineName[]={"MFPolytopeClosestFace"};
  int face;
  int result;
  double t;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFPolytopeClosestFace++;
  starttime=clock();
#endif

  if(P==(MFPolytope)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"First argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeClosestFace+=clock()-starttime;
#endif
    return -1;
   }

  if(s==(MFKVector)NULL)
   {
    sprintf(MFPolytopeErrorMsg,"Second argument is NULL");
    printf("%s -- %s\n",RoutineName,MFPolytopeErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFPolytopeErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFPolytopeClosestFace+=clock()-starttime;
#endif
    return -1;
   }

#ifdef MFTIMINGS
    MFTimeMFPolytopeClosestFace+=clock()-starttime;
#endif
  *d=0.;
  if(P->n==0)return -1;
  result=-1;
  for(face=0;face<P->nFaces;face++)
   {
    t=MFKVDot(s,P->faceN[face])-P->faceO[face];
    if(face==0||t>*d)
     {
      *d=t;
      result=P->face[face];
     }
   }
#ifdef MFTIMINGS
    MFTimeMFPolytopeClosestFace+=clock()-starttime;
#endif
  return result;
 }
