/*
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *      author: Mike Henderson mhender@watson.ibm.com
 *      date:   March 1, 1999
 */

static char *id="@(#) $Id: MFNSpace.c 257 2006-08-30 20:30:07Z mhender $";

#include <multifarioConfig.h>

static char MFNSpaceErrorMsg[256]="";

#include <MFNSpace.h>
#include <MFNVector.h>
#include <MFNKMatrix.h>
#include <MFErrorHandler.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

struct MFNSpaceSt
 {
  int nRefs;
  char *id;
  void *d;

  double (*distance)(MFNSpace,MFNVector,MFNVector,void*,MFErrorHandler);
  double (*inner)(MFNSpace,MFNVector,MFNVector,void*,MFErrorHandler);
  void (*direction)(MFNSpace,MFNVector,MFNVector,MFNVector,void*,MFErrorHandler);
  void (*add)(MFNSpace,MFNVector,MFNVector,MFNVector,void*,MFErrorHandler);
  void (*scale)(MFNSpace,double,MFNVector,MFNVector,void*,MFErrorHandler);
  void (*writedata)(FILE*,MFNSpace,void*,MFErrorHandler);
  void (*freedata)(void*,MFErrorHandler);
 };

void MFRefNSpace(MFNSpace this, MFErrorHandler e)
 {
  static char RoutineName[]={"MFRefNSpace"};

  this->nRefs++;

  return;
 }

void MFFreeNSpace(MFNSpace this, MFErrorHandler e)
 {
  static char RoutineName[]={"MFFreeNSpace"};

  this->nRefs--;
  if(this->nRefs<1)
   {
    if(this->id!=NULL)free(this->id);
    if(this->freedata!=NULL && this->d!=NULL)this->freedata(this->d,e);
    free(this);
   }
  return;
 }

double MFNSpaceDistance(MFNSpace this,MFNVector v0,MFNVector v1, MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceDistance"};
  double result;
  int i;

  result=this->distance(this,v0,v1,this->d,e);

  return result;
 }

void MFNSpaceDirection(MFNSpace this,MFNVector v0,MFNVector v1,MFNVector diff, MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceDirection"};
  double d;
  int i;

  this->direction(this,v0,v1,diff,this->d,e);

  return;
 }

double MFNSpaceInner(MFNSpace this,MFNVector v0,MFNVector v1, MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceInner"};
  double result;
  int i;
  int verbose;

  result=this->inner(this,v0,v1,this->d,e);

  return result;
 }

void MFWriteNSpace(FILE *fid,MFNSpace u, MFErrorHandler e)
 {
  static char RoutineName[]={"MFWriteNSpace"};
  int i;

  fprintf(fid,"%s\n","NSpace");
  fprintf(fid,"%d\n",strlen(u->id));
  fprintf(fid,"%s\n",u->id);
  fprintf(fid,"%d\n",u->nRefs);
  u->writedata(fid,u,u->d,e);

  return;
 }

void MFReadWeightedNSpaceData(FILE *fid,MFNSpace,MFErrorHandler);
#ifdef MFTPBVP
void MFReadTPBVPNSpaceData(FILE *fid,MFNSpace,MFErrorHandler);
#endif
#ifdef HAVE_AUTO
/*void MFReadAUTONSpaceData(FILE *fid,MFNSpace,MFErrorHandler);*/
#endif

MFNSpace MFReadNSpace(FILE *fid, MFErrorHandler e)
 {
  static char RoutineName[]={"MFReadNSpace"};
  int i;
  MFNSpace Omega;
  char tag[100]="";
  int n=0;
  char *id;
  int verbose=0;

  fscanf(fid,"%s\n",tag);
  if(strcmp(tag,"NSpace"))
   {
    sprintf(MFNSpaceErrorMsg,"Next Object is not a NSpace! (%s) -- %s\n",RoutineName,tag);
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return;
   }
  if(verbose)
   {
    printf("%s\n",RoutineName);
    printf("  tag %s\n",tag);fflush(stdout);
   }

  fscanf(fid,"%d\n",&n);
  if(verbose){printf("  len(id) %d\n",n);fflush(stdout);}

  id=malloc((n+1)*sizeof(char));

#ifndef MFNOSAFETYNET
  if(id==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"Out of memory, trying to allocate %d bytes",(n+1)*sizeof(char));
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif

  fscanf(fid,"%s\n",id);
  if(verbose){printf("  id %s\n",id);fflush(stdout);}

  Omega= MFCreateNSpaceBaseClass(id,e);
  fscanf(fid,"%d\n",&(Omega->nRefs));

  if(!strcmp(id,"WeightedNSpace"))
   {
    MFReadWeightedNSpaceData(fid,Omega,e);

#ifdef MFTPBVP
  }else if(!strcmp(id,"TPBVPNSpace"))
   {
    MFReadTPBVPNSpaceData(fid,Omega,e);
#endif

#ifdef HAVE_AUTO
  }else if(!strcmp(id,"AUTONSpace"))
   {
/*  MFReadAUTONSpaceData(fid,Omega,e);*/
#endif

   }else{
    sprintf(MFNSpaceErrorMsg,"Object is not a known NSpace! (%s)\n",RoutineName,tag);
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    free(id);
    return;
   }
  free(id);

  return Omega;
 }

void MFNSpaceAdd(MFNSpace this,MFNVector v0,MFNVector v1,MFNVector sum, MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceAdd"};
  double d;
  int i;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(this->add==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, type %s. add called, but not set.",this->id);
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->add(this,v0,v1,sum,this->d,e);

  return;
 }

double MFNSpaceTangentDistance(MFNSpace this, MFNKMatrix Phi0, MFNKMatrix Phi1, MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceTangentDistance"};
  int i,j,n,k;
  double id[9];
  MFNVector phi0,phi1;

  k=MFNKMatrixK(Phi0,e);
  n=MFNKMatrixN(Phi0,e);

  if(k>2)
   {

#ifdef MFNOCONFIDENCE
    sprintf(MFNSpaceErrorMsg,"Routine only implemented for k<=2, this test will not be used.");
    MFSetError(e,4,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
#endif

    return 0.;
   }


  for(i=0;i<k;i++)
   {
    phi0=MFMColumn(Phi0,i,e);
    for(j=0;j<k;j++)
     {
      phi1=MFMColumn(Phi1,j,e);
      id[i+k*j]=MFNSpaceInner(this,phi0,phi1,e);
      MFFreeNVector(phi1,e);
     }
    MFFreeNVector(phi0,e);
   }

  if(k==1)return fabs(id[0]-1.);
   else if(k==2)return sqrt(fabs(id[0]*id[3]-id[1]*id[2]-1.));

  return 0.;
 }

void MFNSpaceScale(MFNSpace this,double s,MFNVector v,MFNVector prod, MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceScale"};
  double d;
  int i;

#ifdef MFNOCONFIDENCE
    if(this==NULL)
     {
      sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
      MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
      return;
     }

    if(this->add==NULL)
     {
      sprintf(MFNSpaceErrorMsg,"NSpace, type %s. scale called, but not set.",this->id);
      MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
      return;
     }
#endif

  this->scale(this,s,v,prod,this->d,e);

  return;
 }

MFNSpace MFCreateNSpaceBaseClass(char *id, MFErrorHandler e)
 {
  static char RoutineName[]={"MFCreateNSpaceBaseClass"};
  MFNSpace this;

  this=malloc(sizeof(struct MFNSpaceSt));

#ifndef MFNOSAFETYNET
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFNSpaceSt));
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif

  this->id=malloc((strlen(id)+1)*sizeof(char));

#ifndef MFNOSAFETYNET
  if(this->id==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"Out of memory, trying to allocate %d bytes",(strlen(id)+1)*sizeof(char));
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif

  strcpy(this->id,id);
  this->nRefs=1;
  this->d=NULL;
  this->distance=NULL;
  this->inner=NULL;
  this->direction=NULL;
  this->add=NULL;
  this->scale=NULL;
  this->writedata=NULL;
  this->freedata=NULL;

  return this;
 }

void MFNSpaceSetDistance(MFNSpace this,double (*distance)(MFNSpace,MFNVector,MFNVector,void*,MFErrorHandler), MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceSetDistance"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->distance=distance;
  return;
 }

void MFNSpaceSetInnerProduct(MFNSpace this,double (*inner)(MFNSpace,MFNVector,MFNVector,void*,MFErrorHandler), MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceSetDistance"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->inner=inner;
  return;
 }

void MFNSpaceSetDirection(MFNSpace this,void (*direction)(MFNSpace,MFNVector,MFNVector,MFNVector,void*,MFErrorHandler), MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceSetDistance"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->direction=direction;
  return;
 }

void MFNSpaceSetAdd(MFNSpace this,void (*add)(MFNSpace,MFNVector,MFNVector,MFNVector,void*,MFErrorHandler), MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceSetDistance"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->add=add;
  return;
 }

void MFNSpaceSetScale(MFNSpace this,void (*scale)(MFNSpace,double,MFNVector,MFNVector,void*,MFErrorHandler), MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceSetDistance"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->scale=scale;
  return;
 }

void MFNSpaceSetWriteData(MFNSpace this,void (*writedata)(FILE*,MFNSpace,void*,MFErrorHandler), MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceSetDistance"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->writedata=writedata;
  return;
 }

void MFNSpaceSetFreeData(MFNSpace this,void (*freedata)(void *,MFErrorHandler), MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceSetDistance"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->freedata=freedata;
  return;
 }

void MFNSpaceSetData(MFNSpace this,void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceSetDistance"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->d=data;
  return;
 }

void *MFNSpaceGetData(MFNSpace this, MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceGetData"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  return this->d;
 }

char *MFNSpaceGetId(MFNSpace this, MFErrorHandler e)
 {
  static char RoutineName[]={"MFNSpaceGetId"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNSpaceErrorMsg,"NSpace, (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNSpaceErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  return this->id;
 }

