/*
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *      author: Mike Henderson mhender@watson.ibm.com
 *      date:   November 11, 1997
 *              February 2, 1999   Ported to C
 */

static char *id="@(#) $Id: MFNKMatrix.c 257 2006-08-30 20:30:07Z mhender $";

static char MFNKMatrixErrorMsg[256]="";

#include <MFErrorHandler.h>
#include <MFNKMatrix.h>
#include <MFNVector.h>
#include <MFKVector.h>
#include <float.h>
#include <stdlib.h>
#include <string.h>
#include <MFPrint.h>
#include <math.h>

#define DENSE 0
#define VECTOR 1

struct MFNKMatrixSt
 {
  int type;
  int n;
  int k;
  double *data;
  MFNVector *cols;
  int nRefs;
 };

void MFFreeNKMatrix(MFNKMatrix this,MFErrorHandler e)
 {
  static char RoutineName[]={"MFFreeNKMatrix"};
  int i;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Pointer to Matrix (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->nRefs--;

  if(this->nRefs<1)
   {
    if(this->data!=NULL)free(this->data);
    if(this->cols!=NULL)
     {
      for(i=0;i<this->k;i++)MFFreeNVector(this->cols[i],e);
      free(this->cols);
     }
    free(this);
   }
  return;
 }

void MFRefMatrix(MFNKMatrix this,MFErrorHandler e)
 {
  static char RoutineName[]={"MFRefMatrix"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Pointer to Matrix (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->nRefs++;
 }

MFNKMatrix MFCreateNKMatrix(int k,MFNVector *col,MFErrorHandler e)
 {
  static char RoutineName[]={"MFCreateNKMatrix"};
  MFNKMatrix this;
  int i,j;
  double *u;

#ifdef MFNOCONFIDENCE
  if(k<1)
   {
    sprintf(MFNKMatrixErrorMsg,"Number of Columns %d (argument 1) is Illegal. Must be positive.",k);
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return NULL;
   }

  if(col==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"List of Columns (argument 2) is NULL.");
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  this=malloc(sizeof(struct MFNKMatrixSt));

#ifndef MFNOSAFETYNET
  if(this==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFNKMatrixSt));
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif

  this->n=-1;
  this->k=k;

  if(!strcmp(MFNVGetId(col[0],e),"DENSE"))
   {
    this->n=MFNV_NC(col[0],e);
    this->data=malloc(this->n*this->k*sizeof(double));

#ifndef MFNOSAFETYNET
    if(this->data==NULL)
     {
      sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",this->n*this->k*sizeof(double));
      MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
      MFErrorHandlerOutOfMemory(e);
      free(this);
      return NULL;
     }
#endif

    this->cols=NULL;

    for(j=0;j<k;j++)
     {
      u=MFNV_CStar(col[j],e);
      for(i=0;i<this->n;i++)
       {
        this->data[i+this->n*j]=u[i];
       }
     }
    this->type=DENSE;
   }else{
    this->data=NULL;

    this->cols=malloc(this->k*sizeof(MFNVector));

#ifndef MFNOSAFETYNET
    if(this->cols==NULL)
     {
      sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",this->k*sizeof(MFNVector));
      MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
      MFErrorHandlerOutOfMemory(e);
      free(this);
      return NULL;
     }
#endif

    for(j=0;j<k;j++)
     {
      this->cols[j]=col[j];
      MFRefNVector(col[j],e);
     }

    if(!strcmp(MFNVGetId(col[0],e),"DENSE"))this->type=DENSE;
     else this->type=VECTOR;
   }

  this->nRefs=1;

  return this;
 }

MFNKMatrix MFCreateNKMatrixWithData(int n,int k, double *data,MFErrorHandler e)
 {
  static char RoutineName[]={"MFCreateNKMatrixWithData"};
  MFNKMatrix this;
  int i,j;

#ifdef MFNOCONFIDENCE
  if(n<1)
   {
    sprintf(MFNKMatrixErrorMsg,"Number of Rows %d (argument 1) is Illegal. Must be positive.",n);
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return NULL;
   }

  if(k<1)
   {
    sprintf(MFNKMatrixErrorMsg,"Number of Columns %d (argument 2) is Illegal. Must be positive.",k);
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  this=malloc(sizeof(struct MFNKMatrixSt));

#ifndef MFNOSAFETYNET
  if(this==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFNKMatrixSt));
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif

  this->n=n;
  this->k=k;

  this->data=malloc(this->n*this->k*sizeof(double));

#ifndef MFNOSAFETYNET
  if(this->data==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",this->n*this->k*sizeof(double));
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    free(this);
    return NULL;
   }
#endif

  this->nRefs=1;

  if(data!=NULL)
   {
    for(j=0;j<k;j++)
     {
      for(i=0;i<n;i++)
       {
        this->data[i+n*j]=data[i+n*j];
       }
     }
   }else{
    for(i=0;i<n*k;i++)this->data[i]=0.;
   }

  this->cols=NULL;
  this->type=DENSE;

  return this;
 }

void MFMVMul(MFNSpace R, MFNKMatrix this,MFKVector s,MFNVector u,MFErrorHandler e)
 {
  static char RoutineName[]={"MFMVMul"};
  int i,j;
  MFNVector col;
  MFNVector v;
  double t;
  static int verbose=0;

#ifdef MFALLOWVERBOSE
  if(verbose){printf("%s\n",RoutineName);fflush(stdout);}
#endif

/* Need to use Add and Scale from NSpace */
  v=MFCloneNVector(u,e);

  for(i=0;i<this->k;i++)
   {
    if(this->type==DENSE)
      col=MFCreateNVectorWithData(this->n,this->data+i*this->n,e);
     else
      col=MFMColumn(this,i,e);

#ifdef MFNOCONFIDENCE
    if(col==NULL)
     {
      sprintf(MFNKMatrixErrorMsg,"col %d of this is NULL");
      MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
      return;
     }
#endif

    t=MFKV_C(s,i,e);
    if(i==0)
     {
      MFNSpaceScale(R,t,col,u,e);
     }else{
      MFNSpaceScale(R,t,col,v,e);
      MFNSpaceAdd(R,u,v,u,e);
     }
    MFFreeNVector(col,e);
   }
  MFFreeNVector(v,e);
  return;
 }

void MFMVMulT(MFNSpace R, MFNKMatrix this,MFNVector u,MFKVector s,MFErrorHandler e)
 {
  static char RoutineName[]={"MFMVMulT"};
  int i,j;
  double t;
  MFNVector col;
  int verbose=0;

#ifdef MFALLOWVERBOSE
  if(verbose){printf("%s, n=%d\n",RoutineName,this->n);fflush(stdout);}
#endif

  for(j=0;j<this->k;j++)
   {
    col=MFMColumn(this,j,e);
    t=MFNSpaceInner(R,col,u,e);

#ifdef MFALLOWVERBOSE
    if(verbose)
     {
      printf("   s[%d]=",j);MFPrintNVector(stdout,col,e);
      printf(".");MFPrintNVector(stdout,u,e);
      printf("=%lf\n",t);
     }
#endif

    MFKVSetC(s,j,t,e);
    MFFreeNVector(col,e);
   }

  return;
 }

double *MFNKM_CStar(MFNKMatrix this,MFErrorHandler e)
 {
  static char RoutineName[]={"MFNKM_CStar"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Pointer to Matrix (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return NULL;
   }

  if(this->type!=DENSE)
   {
    sprintf(MFNKMatrixErrorMsg,"Trying to get dense arrays from non-dense Vector type %d",this->type);
    printf("%s\n",MFNKMatrixErrorMsg);fflush(stdout);
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  return this->data;
 }

MFNVector MFMColumn(MFNKMatrix this,int j,MFErrorHandler e)
 {
  static char RoutineName[]={"MFMColumn"};
  int i;
  MFNVector c;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Pointer to Matrix (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return NULL;
   }

  if(j>this->k)
   {
    sprintf(MFNKMatrixErrorMsg,"Request for invalid column, %d must be in [0,%d)",j,this->k);
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  switch(this->type)
   {
    case DENSE:
     c=MFCreateWrappedNVector(this->n,this->data+j*this->n,e);
     break;
    default:
     c=this->cols[j];
     MFRefNVector(c,e);
   }

  return c;
 }

void MFMRow(MFNKMatrix this,int i,MFKVector r,MFErrorHandler e)
 {
  static char RoutineName[]={"MFMRow"};
  int j;

  switch(this->type)
   {
    case DENSE:
     for(j=0;j<this->k;j++)
       MFKVSetC(r,j,this->data[i+this->n*j],e);
     break;
    default:
     for(j=0;j<this->k;j++)
       MFKVSetC(r,j,MFNV_C(this->cols[j],i,e),e);
     break;
   }

  return;
 }

int MFNKMatrixK(MFNKMatrix this,MFErrorHandler e)
 {
  return this->k;
 }

int MFNKMatrixN(MFNKMatrix this,MFErrorHandler e)
 {
  return this->n;
 }

void MFWriteNKMatrix(FILE *fid,MFNKMatrix L,MFErrorHandler e)
 {
  static char RoutineName[]={"MFWriteNKMatrix"};
  int i;

  fprintf(fid,"%s\n","NKMatrix");
  fprintf(fid,"%d %d %d %d\n",L->type,L->n,L->k,L->nRefs);
  if(L->type==DENSE)
   {
    for(i=0;i<L->n*L->k;i++)
     {
      if(i>0)fprintf(fid," ");
      fprintf(fid,"%lf",L->data[i]);
     }
    fprintf(fid,"\n");
   }else{
    for(i=0;i<L->k;i++)
      MFWriteNVector(fid,L->cols[i],e);
   }

  return;
 }

MFNKMatrix MFReadNKMatrix(FILE *fid,MFErrorHandler e)
 {
  static char RoutineName[]={"MFReadNKMatrix"};
  int i;
  MFNKMatrix L;
  char tag[100]="";

  fscanf(fid,"%s\n",tag);

#ifdef MFNOCONFIDENCE
  if(strcmp(tag,"NKMatrix"))
   {
    sprintf(MFNKMatrixErrorMsg,"Next Object is not a NKMatrix! (%s)\n",RoutineName,tag);
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  L=malloc(sizeof(struct MFNKMatrixSt));

#ifndef MFNOSAFETYNET
  if(L==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFNKMatrixSt));
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif

  fscanf(fid,"%d %d %d %d\n",&(L->type),&(L->n),&(L->k),&(L->nRefs));

  if(L->type==DENSE)
   {
    L->data=malloc(L->n*L->k*sizeof(double));

#ifndef MFNOSAFETYNET
    if(L->data==NULL)
     {
      sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",L->n*L->k*sizeof(double));
      MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
      MFErrorHandlerOutOfMemory(e);
      return NULL;
     }
#endif

    for(i=0;i<L->n*L->k;i++)
     {
      if(i>0)fscanf(fid," ");
      fscanf(fid,"%lf",&(L->data[i]));
     }
    fscanf(fid,"\n");
    L->cols=NULL;
   }else{
    L->cols=malloc(L->k*sizeof(MFNVector));

#ifndef MFNOSAFETYNET
    if(L->cols==NULL)
     {
      sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",L->k*sizeof(MFNVector));
      MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
      MFErrorHandlerOutOfMemory(e);
      free(L);
      return NULL;
     }
#endif

    for(i=0;i<L->k;i++)
      L->cols[i]=MFReadNVector(fid,e);
    L->data=NULL;
   }

  return L;
 }

void MFNKMSetC(MFNKMatrix A,int i,int j,double Aij,MFErrorHandler e)
 {
  static char RoutineName[]={"MFNKMSetC"};

#ifdef MFNOCONFIDENCE
  if(A==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Pointer to Matrix (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(i<0)
   {
    sprintf(MFNKMatrixErrorMsg,"column index (argument 2) is less than zero");
    MFSetError(e,4,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }
  if(i>=A->n)
   {
    sprintf(MFNKMatrixErrorMsg,"column index (argument 2) is too large must be < %d)",A->n);
    MFSetError(e,4,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(j<0)
   {
    sprintf(MFNKMatrixErrorMsg,"row index (argument 3) is less than zero");
    MFSetError(e,4,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }
  if(j>=A->k)
   {
    sprintf(MFNKMatrixErrorMsg,"row index (argument 3) is too large must be < %d)",A->k);
    MFSetError(e,4,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  switch(A->type)
   {
    case DENSE:
     A->data[i+A->n*j]=Aij;
     break;
    default:
     MFNVSetC(A->cols[j],i,Aij,e);
     break;
   }

  return;
 }

void MFRefNKMatrix(MFNKMatrix this,MFErrorHandler e)
 {
  static char RoutineName[]={"MFRefNKMatrix"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Pointer to Matrix (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->nRefs++;

  return;
 }

void MFGramSchmidt(MFNSpace space,MFNKMatrix A,MFErrorHandler e)
 {
  static char RoutineName[]={"MFGramSchmidt"};
  double *a;
  int i,j,n;
  int k;
  MFNVector u;
  MFNVector v;
  MFNVector w;
  double inner;
  int verbose=0;

  n=MFNKMatrixN(A,e);
  k=MFNKMatrixK(A,e);

  switch(A->type)
   {
    case DENSE:
     a=MFNKM_CStar(A,e);
     MFGramSchmidtNoMat(n,k,a,e);
    default:
     u=MFMColumn(A,0,e);
     w=MFCloneNVector(u,e);
     MFFreeNVector(u,e);

#ifdef MFALLOWVERBOSE
     if(verbose){printf("Gram Schmidt:\n");fflush(stdout);}
#endif

     for(i=0;i<k;i++)
      {
       u=MFMColumn(A,i,e);

       inner=MFNSpaceInner(space,u,u,e);

#ifdef MFALLOWVERBOSE
       if(verbose){printf("u_%d\n",i);MFPrintNVector(stdout,u,e);printf("\n");fflush(stdout);
                   printf("<u_%d,u_%d>=%lf\n",i,i,inner);fflush(stdout);}
#endif

       inner=1./sqrt(inner);
       MFNSpaceScale(space,inner,u,u,e);

#ifdef MFALLOWVERBOSE
       if(verbose){printf("normalized u_%d\n",i);MFPrintNVector(stdout,u,e);printf("\n");fflush(stdout);}
#endif
   
       for(j=i+1;j<k;j++)
        {
         v=MFMColumn(A,j,e);

#ifdef MFALLOWVERBOSE
         if(verbose){printf("u_%d\n",j);MFPrintNVector(stdout,v,e);printf("\n");fflush(stdout);}
#endif
   
         inner=MFNSpaceInner(space,u,v,e);

#ifdef MFALLOWVERBOSE
         if(verbose){printf("<u_%d,u_%d>=%lf\n",i,j,inner);fflush(stdout);}
#endif
   
         MFNSpaceScale(space,inner,u,w,e);

#ifdef MFALLOWVERBOSE
         if(verbose){printf("<u_%d,u_%d>u_%d\n",j,i,i);MFPrintNVector(stdout,w,e);printf("\n");fflush(stdout);}
#endif
   
         MFNVDiff(v,w,v,e);

#ifdef MFALLOWVERBOSE
         if(verbose){printf("u_%d-<u_%d,u_%d>u_%d\n",j,j,i,i);MFPrintNVector(stdout,v,e);printf("\n");fflush(stdout);}
#endif
   
         MFFreeNVector(v,e);
        }
       MFFreeNVector(u,e);
      }
     MFFreeNVector(w,e);
   }

  return;
 }

void MFGramSchmidtNoMat(int n, int k, double *a,MFErrorHandler e)
 {
  static char RoutineName[]={"MFGramSchmidtNoMat"};
  double inner;
  int i,j,l;
  double *u;
  double *v;
  int verbose=0;

#ifdef MFALLOWVERBOSE
  if(verbose){printf("%s\n",RoutineName);fflush(stdout);}
#endif

  for(i=0;i<k;i++)
   {
    u=a+n*i;

    inner=0.;for(l=0;l<n;l++)inner+=u[l]*u[l];
    inner=1./sqrt(inner);
    for(l=0;l<n;l++)u[l]=u[l]*inner;

    for(j=i+1;j<k;j++)
     {
      v=a+n*j;
      inner=0.;for(l=0;l<n;l++)inner+=u[l]*v[l];
      for(l=0;l<n;l++)v[l]=v[l]-inner*u[l];
     }
   }

  return;
 }

void MFNKMProjectTangentForBranchSwitch(MFNSpace space, MFNKMatrix A,MFNVector phi,MFNKMatrix Phi,MFErrorHandler e)
 {
  static char RoutineName[]={"MFNKMProjectTangentForBranchSwitch"};

  int i,j;
  int k,m;
  MFNVector u,w;
  double *p;
  double pmax;
  MFNVector pi,pj;
  int verbose=0;

#ifdef MFALLOWVERBOSE
  if(verbose){printf("%s\n",RoutineName);fflush(stdout);
              printf("  A=");MFPrintNKMatrix(stdout,A,e);fflush(stdout);
              printf("  phi=");MFPrintNVector(stdout,phi,e);printf("\n");fflush(stdout);}
#endif

  k=MFNKMatrixK(A,e);

  p=malloc(k*sizeof(double));

#ifndef MFNOSAFETYNET
  if(p==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",k*sizeof(double));
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return;
   }
#endif

  j=0;
  pmax=0.;
  for(i=0;i<k;i++)
   {
    u=MFMColumn(A,i,e);
    p[i]=MFNSpaceInner(space,phi,u,e);
    if(fabs(p[i])>pmax){pmax=fabs(p[i]);j=i;}
    MFFreeNVector(u,e);

#ifdef MFALLOWVERBOSE
    if(verbose){printf("phi^* A_[%d]=%lf\n",i,p[i]);fflush(stdout);}
#endif

   }

#ifdef MFALLOWVERBOSE
  if(verbose){printf("max of phi^* A_j is for j=%d, =%lf\n",j,pmax);fflush(stdout);}
#endif


  m=0;
  u=MFMColumn(Phi,m,e);
  w=MFCloneNVector(u,e);
  MFNSpaceScale(space,1.,phi,u,e);

#ifdef MFALLOWVERBOSE
  if(verbose){printf("column 0 of Phi is phi ");MFPrintNVector(stdout,u,e);printf("\n");(stdout);}
#endif

  if(Phi->type==DENSE)MFMSetColumn(Phi,m,u,e);
  MFFreeNVector(u,e);
  m++;

  pj=MFMColumn(A,j,e);
  for(i=0;i<k;i++)
   {
    if(i!=j)
     {
      pi=MFMColumn(A,i,e);
      u=MFMColumn(Phi,m,e);

      MFNSpaceScale(space, p[i],pj,u,e);
      MFNSpaceScale(space,-p[j],pi,w,e);
      MFNSpaceAdd(space,u,w,u,e);

#ifdef MFALLOWVERBOSE
      if(verbose){printf("column %d of Phi is %lf A_%d - %lf A_%d ",m,p[i],j,p[j],i);MFPrintNVector(stdout,u,e);printf("\n");(stdout);}
#endif

      if(Phi->type==DENSE)MFMSetColumn(Phi,m,u,e);
      m++;
      MFFreeNVector(u,e);
      MFFreeNVector(pi,e);
     }
   }
  MFFreeNVector(pj,e);
  free(p);

#ifdef MFALLOWVERBOSE
  if(verbose){printf("Before G-S:\n");MFPrintNKMatrix(stdout,Phi,e);fflush(stdout);}
#endif

  MFGramSchmidt(space,Phi,e);


#ifdef MFALLOWVERBOSE
  if(verbose){printf("After G-S:\n");MFPrintNKMatrix(stdout,Phi,e);fflush(stdout);
              printf("done %s\n",RoutineName);fflush(stdout);}
#endif

  return;
 }

MFNKMatrix MFCloneNKMatrix(MFNKMatrix A,MFErrorHandler e)
 {
  static char RoutineName[]={"MFCloneNKMatrix"};
  MFNKMatrix this;
  int i,j;
  double *u;

#ifdef MFNOCONFIDENCE
  if(A==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Cloning a NULL Matrix!");
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  this=malloc(sizeof(struct MFNKMatrixSt));

#ifndef MFNOSAFETYNET
  if(this==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFNKMatrixSt));
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif

  this->n=A->n;
  this->k=A->k;
  this->type=A->type;
  if(this->type==DENSE)
   {
    this->data=malloc(this->n*this->k*sizeof(double));

#ifndef MFNOSAFETYNET
    if(this->data==NULL)
     {
      sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",this->n*this->k*sizeof(double));
      MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
      MFErrorHandlerOutOfMemory(e);
      free(this);
      return NULL;
     }
#endif

    this->cols=NULL;

    for(j=0;j<this->k;j++)
     {
      for(i=0;i<this->n;i++)this->data[i+this->n*j]=A->data[i+this->n*j];
     }
   }else{
    this->data=NULL;

    this->cols=malloc(this->k*sizeof(MFNVector));

#ifndef MFNOSAFETYNET
    if(this->cols==NULL)
     {
      sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",this->k*sizeof(MFNVector));
      MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
      MFErrorHandlerOutOfMemory(e);
      free(this);
      return NULL;
     }
#endif

    for(j=0;j<this->k;j++)
      this->cols[j]=MFCloneNVector(A->cols[j],e);
   }

  this->nRefs=1;

  return this;
 }

void MFMSetColumn(MFNKMatrix this,int j, MFNVector u,MFErrorHandler e)
 {
  static char RoutineName[]={"MFMSetColumn"};
  int i;
  MFNVector c;

  switch(this->type)
   {
    case DENSE:
     for(i=0;i<this->n;i++)
      this->data[i+this->n*j]=MFNV_C(u,i,e);
     break;
    case VECTOR:
      MFRefNVector(u,e);
      MFFreeNVector(this->cols[j],e);
      this->cols[j]=u;
     break;
    default:
     for(i=0;i<this->n;i++)
      MFNVSetC(this->cols[j],i,MFNV_C(u,i,e),e);
   }

  return;
 }

void MFGramSchmidtReplace(MFNSpace space,MFNKMatrix A,MFNVector phi0, MFNVector phi1,MFErrorHandler e)
 {
  static char RoutineName[]={"MFGramSchmidtReplace"};
  double *a;
  int i,j,n;
  int k;
  MFNVector u;
  MFNVector v;
  MFNVector w;
  double inner;
  int zero;
  double *B=NULL;
  int sign0,sign1;
  int verbose=0;

/* Remove vector u from the basis, and replaces it with the vector v */
  if(verbose){printf("%s\n",RoutineName);fflush(stdout);}

  n=MFNKMatrixN(A,e);
  k=MFNKMatrixK(A,e);

  B=malloc(k*k*sizeof(double));

#ifndef MFNOSAFETYNET
    if(B==NULL)
     {
      sprintf(MFNKMatrixErrorMsg,"Out of memory, trying to allocate %d bytes",k*k*sizeof(double));
      MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
      MFErrorHandlerOutOfMemory(e);
      return;
     }
#endif

  for(i=0;i<k;i++)
   {
    u=MFMColumn(A,i,e);
    for(j=0;j<k;j++)
     {
      v=MFMColumn(A,j,e);
      B[i+k*j]=MFNSpaceInner(space,u,v,e);
      MFFreeNVector(v,e);
     }
    MFFreeNVector(u,e);
   }
  sign0=MFAtlasDet(k,B,e)>0;

  u=MFMColumn(A,0,e);
  w=MFCloneNVector(u,e);
  MFFreeNVector(u,e);

#ifdef MFALLOWVERBOSE
  if(verbose){printf("Gram Schmidt:\n");fflush(stdout);}
#endif

  zero=-1;
  for(i=-1;i<k;i++)
   {
    if(i>-1)u=MFMColumn(A,i,e);
     else{u=phi0;MFRefNVector(phi0,e);}

    inner=MFNSpaceInner(space,u,u,e);

#ifdef MFALLOWVERBOSE
    if(verbose){printf("u_%d\n",i);MFPrintNVector(stdout,u,e);printf("\n");fflush(stdout);
                printf("<u_%d,u_%d>=%lf\n",i,i,inner);fflush(stdout);}
#endif

    if(inner>1.e-7)
     {
      inner=1./sqrt(inner);
      MFNSpaceScale(space,inner,u,u,e);

#ifdef MFALLOWVERBOSE
      if(verbose){printf("normalized u_%d\n",i);MFPrintNVector(stdout,u,e);printf("\n");fflush(stdout);}
#endif

     }else{
      zero=i;

#ifdef MFALLOWVERBOSE
      if(verbose){printf("zero vector u_%d\n",i);MFPrintNVector(stdout,u,e);printf("\n");fflush(stdout);}
#endif

     }
 
    for(j=i+1;j<k;j++)
     {
      v=MFMColumn(A,j,e);

#ifdef MFALLOWVERBOSE
      if(verbose){printf("u_%d\n",j);MFPrintNVector(stdout,v,e);printf("\n");fflush(stdout);}
#endif

      inner=MFNSpaceInner(space,u,v,e);

#ifdef MFALLOWVERBOSE
      if(verbose){printf("<u_%d,u_%d>=%lf\n",i,j,inner);fflush(stdout);}
#endif

      MFNSpaceScale(space,inner,u,w,e);

#ifdef MFALLOWVERBOSE
      if(verbose){printf("<u_%d,u_%d>u_%d\n",j,i,i);MFPrintNVector(stdout,w,e);printf("\n");fflush(stdout);}
#endif

      MFNVDiff(v,w,v,e);

#ifdef MFALLOWVERBOSE
      if(verbose){printf("u_%d-<u_%d,u_%d>u_%d\n",j,j,i,i);MFPrintNVector(stdout,v,e);printf("\n");fflush(stdout);}
#endif

      MFFreeNVector(v,e);
     }
    MFFreeNVector(u,e);

#ifdef MFALLOWVERBOSE
    if(verbose){printf("\n");fflush(stdout);}
#endif

   }
  MFFreeNVector(w,e);

/* Delete the zero column (move it to the first position) */

  for(i=zero;i>-1;i--)
   {
    u=MFMColumn(A,i-1,e);
    v=MFMColumn(A,i,e);
    MFMSetColumn(A,i,u,e);
    MFFreeNVector(u,e);
    MFFreeNVector(v,e);
   }

/* Replace the zero column with phi1 */

  u=MFMColumn(A,0,e);
  MFFreeNVector(u,e);
  MFMSetColumn(A,0,phi1,e);

 MFGramSchmidt(space,A,e);

  for(i=0;i<k;i++)
   {
    u=MFMColumn(A,i,e);
    for(j=0;j<k;j++)
     {
      v=MFMColumn(A,j,e);
      B[i+k*j]=MFNSpaceInner(space,u,v,e);
      MFFreeNVector(v,e);
     }
    MFFreeNVector(u,e);
   }
  sign1=MFAtlasDet(k,B,e)>0;

  if(sign0&&!sign1||sign1&&!sign0)
   {
    u=MFMColumn(A,0,e);
    MFNSpaceScale(space,-1.,u,u,e);
    MFFreeNVector(u,e);
   }

  free(B);
  if(verbose){printf("done %s\n",RoutineName);fflush(stdout);}

  return;
 }

void MFMMMul(MFNSpace R, MFNKMatrix Phi0,MFNKMatrix Phi1,double *prod,MFErrorHandler e)
 {
  static char RoutineName[]={"MFMMMul"};
  int i,j;
  MFNVector col0,col1;
  static int verbose;

/* Phi_0^T Phi_1 */

  verbose=0;
  if(verbose){printf("%s\n",RoutineName);fflush(stdout);}

#ifdef MFNOCONFIDENCE
  if(Phi0->n!=Phi1->n)
   {
    sprintf(MFNKMatrixErrorMsg,"Can't do product of incompatible matrices. A.n=%d, B.n=%d",Phi0->n,Phi1->n);
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(Phi0->type!=Phi1->type)
   {
    sprintf(MFNKMatrixErrorMsg,"Can't do product of incompatible matrices. A.type=%d, B.type=%d",Phi0->type,Phi1->type);
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  for(i=0;i<Phi0->k;i++)
   {
    if(Phi0->type==DENSE)
      col0=MFCreateNVectorWithData(Phi0->n,Phi0->data+i*Phi0->n,e);
     else
      col0=MFMColumn(Phi0,i,e);

#ifdef MFNOCONFIDENCE
    if(col0==NULL)
     {
      sprintf(MFNKMatrixErrorMsg,"col %d of A is NULL",i);
      MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
      return;
     }
#endif

    for(j=0;j<Phi1->k;j++)
     {
      if(Phi1->type==DENSE)
        col1=MFCreateNVectorWithData(Phi1->n,Phi1->data+i*Phi1->n,e);
       else
        col1=MFMColumn(Phi1,j,e);
  
#ifdef MFNOCONFIDENCE
      if(col1==NULL)
       {
        sprintf(MFNKMatrixErrorMsg,"col %d of B is NULL",j);
        MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
        return;
       }
#endif

      prod[i+Phi0->k*j]=MFNSpaceInner(R,col0,col1,e);
      MFFreeNVector(col1,e);
     }
    MFFreeNVector(col0,e);
   }
  return;
 }

void MFPrintNKMatrix(FILE *fid,MFNKMatrix L,MFErrorHandler e)
 {
  static char RoutineName[]={"MFPrintNKMatrix"};
  int i,n,k;
  MFKVector s;

#ifdef MFNOCONFIDENCE
  if(fid==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"fid (argument 1) is NULL.");
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(L==NULL)
   {
    sprintf(MFNKMatrixErrorMsg,"Matrix (argument 2) is NULL.");
    MFSetError(e,12,RoutineName,MFNKMatrixErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  n=MFNKMatrixN(L,e);
  k=MFNKMatrixK(L,e);
  s=MFCreateKVector(k,e);
  if(L->type==VECTOR)
   {
    printf("vector\n");fflush(stdout);
    printf("[\n");
    for(i=0;i<k;i++)
     {
      MFPrintNVector(fid,L->cols[i],e);
      fprintf(fid,"\n");fflush(stdout);
     }
    printf("]^T\n");
   }else if(n<16)
   {
    for(i=0;i<n;i++)
     {
      MFMRow(L,i,s,e);
      MFPrintKVector(fid,s,e);
      fprintf(fid,"\n");
     }
   }else{
    printf("dense\n");fflush(stdout);
    for(i=0;i<8;i++)
     {
      MFMRow(L,i,s,e);
      MFPrintKVector(fid,s,e);
      fprintf(fid,"\n");
     }
    fprintf(fid,"...\n");
    for(i=n-8;i<n;i++)
     {
      MFMRow(L,i,s,e);
      MFPrintKVector(fid,s,e);
      fprintf(fid,"\n");
     }
   }
  MFFreeKVector(s,e);
  printf("done print matrix\n");fflush(stdout);
  return;
 }
