/*
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *      author: Mike Henderson mhender@watson.ibm.com
 *      date:   November 11, 1997
 *              February 2, 1999   Ported to C
 */

static char *id="@(#) $Id: MFLOCANVector.c 257 2006-08-30 20:30:07Z mhender $";

#include <MFErrorHandler.h>
#include <MFNVector.h>
#include <float.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <MFTime.h>

struct MFLOCANVectorData
 {
  int nx;
  double *x;
  int np;
  double *p;
 };

void MFLOCANVDiff(void*,void*,void*);
void MFLOCANVAdd(void*,void*,void*);
void MFWriteLOCANVectorData(FILE*,void*);
void MFReadLOCANVectorData(FILE*,MFNVector);
void MFPrintLOCANVector(FILE*,void*);
MFNVector MFCloneLOCANVector(void*);
MFNVector MFCreateLOCANVectorWithData(int,double*,int,double*);
MFNVector MFCreateLOCANVector(int,int);
void MFPrintLOCANVector(FILE*,void*);

double *alloc_vec();
void free_vec(double**);

static char MFNVectorErrorMsg[256]="";

void MFLOCANVDiff(void *adata,void *bdata, void *cdata)
 {
  static char RoutineName[]={"MFLOCANVDiff"};
  int i,n;
  struct MFLOCANVectorData *a;
  struct MFLOCANVectorData *b;
  struct MFLOCANVectorData *c;

  a=(struct MFLOCANVectorData*)adata;
  if(a==(struct MFLOCANVectorData*)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for a (argument 1) is NULL");
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  b=(struct MFLOCANVectorData*)bdata;
  if(b==(struct MFLOCANVectorData*)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for b (argument 2) is NULL");
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  c=(struct MFLOCANVectorData*)cdata;
  if(c==(struct MFLOCANVectorData*)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for c (argument 3) is NULL");
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(a->nx!=b->nx || a->nx!=c->nx || b->nx!=c->nx)
   {
    sprintf(MFNVectorErrorMsg,"Vectors must all be the same length a=%d, b=%d, c=%d",a->nx,b->nx,c->nx);
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);
    MFSetError(4,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFNVDiff+=clock()-starttime;
#endif
    return;
   }

  for(i=0;i<a->nx;i++)c->x[i]=a->x[i]-b->x[i];
  for(i=0;i<a->np;i++)c->p[i]=a->p[i]-b->p[i];

  return;
 }

void MFLOCANVAdd(void *adata,void *bdata,void *cdata)
 {
  static char RoutineName[]={"MFLOCANVAdd"};
  int i,n;
  struct MFLOCANVectorData *a;
  struct MFLOCANVectorData *b;
  struct MFLOCANVectorData *c;

  a=(struct MFLOCANVectorData*)adata;
  if(a==(struct MFLOCANVectorData*)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for a (argument 1) is NULL");
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  b=(struct MFLOCANVectorData*)bdata;
  if(b==(struct MFLOCANVectorData*)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for b (argument 2) is NULL");
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  c=(struct MFLOCANVectorData*)cdata;
  if(c==(struct MFLOCANVectorData*)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for c (argument 3) is NULL");
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(a->nx!=b->nx || a->nx!=c->nx || b->nx!=c->nx)
   {
    sprintf(MFNVectorErrorMsg,"Vectors must all be the same length a=%d, b=%d, c=%d",a->nx,b->nx,c->nx);
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);
    MFSetError(4,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFNVDiff+=clock()-starttime;
#endif
    return;
   }

  for(i=0;i<a->nx;i++)c->x[i]=a->x[i]+b->x[i];
  for(i=0;i<a->np;i++)c->p[i]=a->p[i]+b->p[i];

  return;
 }

MFNVector MFCloneLOCANVector(void *data)
 {
  static char RoutineName[]={"MFCloneLOCANVector"};
  struct MFLOCANVectorData *u;

  u=(struct MFLOCANVectorData*)data;

  return MFCreateLOCANVectorWithData(u->nx,u->x,u->np,u->p);
 }

MFNVector MFCreateLOCANVectorWithData(int nx,double *x,int np, double *p)
 {
  static char RoutineName[]={"MFCreateLOCANVector"};
  MFNVector this;
  struct MFLOCANVectorData *data;
  int i;

#ifdef MFTIMINGS
  clock_t starttime;

  MFCalledMFCreateNVector++;
  starttime=clock();
#endif

  if(nx<1||np<1)
   {
    sprintf(MFNVectorErrorMsg,"Length of Vector %d and %d (argument 1 and 3) is Illegal. Must be positive.",nx,np);
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateNVector+=clock()-starttime;
#endif
    return((MFNVector)NULL);
   }

  data=malloc(sizeof(struct MFLOCANVectorData)); /*done*/
  if(data==(struct MFLOCANVectorData*)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFLOCANVectorData));
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return(this);
   }

  data->nx=nx;
  data->x=alloc_vec();
  vec_copy(x,data->x);

  data->np=np;
  data->p=malloc(np*sizeof(double)); /*done*/
  if(data->p==(double*)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Out of memory, trying to allocate %d bytes",np*sizeof(double));
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
#ifdef MFTIMINGS
    MFTimeMFCreateNVector+=clock()-starttime;
#endif
    return (MFNVector)NULL;
   }

  if(nx>0&&x!=(double*)NULL)
    for(i=0;i<nx;i++)(data->x)[i]=x[i];
  if(np>0&&p!=(double*)NULL)
    for(i=0;i<np;i++)data->p[i]=p[i];

  this=MFCreateNVectorBaseClass("LOCA");

  MFNVectorSetData(this,data);

  MFNVectorSetDiff(this,MFLOCANVDiff);
  MFNVectorSetAdd(this,MFLOCANVAdd);
  MFNVectorSetClone(this,MFCloneLOCANVector);
  MFNVectorSetPrint(this,MFPrintLOCANVector);

#ifdef MFTIMINGS
    MFTimeMFCreateNVector+=clock()-starttime;
#endif
  return this;
 }

MFNVector MFCreateLOCANVector(int nx,int np)
 {
  static char RoutineName[]={"MFCreateLOCANVector"};
  MFNVector this;

  return MFCreateLOCANVectorWithData(nx,(double*)NULL,np,(double*)NULL);
 }

double *MFLOCANVP(MFNVector this)
 {
  static char RoutineName[]={"MFLOCANVP"};
  struct MFLOCANVectorData *u;

  if(this==(MFNVector)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector (argument 1) is NULL");
    MFSetError(4,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return (double*)NULL;
   }

  if(strcmp(MFNVGetId(this),"LOCA"))
   {
    sprintf(MFNVectorErrorMsg,"Trying to get dense arrays from non-dense Vector type %d",MFNVGetId(this));
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return (double*)NULL;
   }

  u=(struct MFLOCANVectorData*)MFNVectorGetData(this);

  return(u->p);
 }

double *MFLOCANVX(MFNVector this)
 {
  static char RoutineName[]={"MFLOCANVX"};
  struct MFLOCANVectorData *u;

  if(this==(MFNVector)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector (argument 1) is NULL");
    MFSetError(4,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return (double*)NULL;
   }

  if(strcmp(MFNVGetId(this),"LOCA"))
   {
    sprintf(MFNVectorErrorMsg,"Trying to get dense arrays from non-dense Vector type %d",MFNVGetId(this));
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return (double*)NULL;
   }

  u=(struct MFLOCANVectorData*)MFNVectorGetData(this);

  return(u->x);
 }

int MFLOCANVNX(MFNVector this)
 {
  static char RoutineName[]={"MFLOCANVNX"};
  struct MFLOCANVectorData *u;

  if(this==(MFNVector)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector (argument 1) is NULL");
    MFSetError(4,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return -1;
   }

  if(strcmp(MFNVGetId(this),"LOCA"))
   {
    sprintf(MFNVectorErrorMsg,"Trying to get dense arrays from non-dense Vector type %d",MFNVGetId(this));
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return -1;
   }

  u=(struct MFLOCANVectorData*)MFNVectorGetData(this);

  return(u->nx);
 }

int MFLOCANVNP(MFNVector this)
 {
  static char RoutineName[]={"MFLOCANVNP"};
  struct MFLOCANVectorData *u;

  if(this==(MFNVector)NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector (argument 1) is NULL");
    MFSetError(4,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return -1;
   }

  if(strcmp(MFNVGetId(this),"LOCA"))
   {
    sprintf(MFNVectorErrorMsg,"Trying to get dense arrays from non-dense Vector type %d",MFNVGetId(this));
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return -1;
   }

  u=(struct MFLOCANVectorData*)MFNVectorGetData(this);

  return(u->np);
 }

void MFPrintLOCANVector(FILE *fid,void *data)
 {
  static char RoutineName[]={"MFPrintLOCANVector"};
  double unorm;
  int i;
  struct MFLOCANVectorData *u;

  u=(struct MFLOCANVectorData*)data;

  if(fid==(FILE*)NULL)
   {
    sprintf(MFNVectorErrorMsg,"fid (argument 1) is NULL.");
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(data==(void*)NULL)
   {
    sprintf(MFNVectorErrorMsg,"data (argument 2) is NULL.");
    printf("%s -- %s\n",RoutineName,MFNVectorErrorMsg);fflush(stdout);
    MFSetError(12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  unorm=MFPrintMetricLOCA(u->x,u->p);
  fprintf(fid,"LOCA(...,%lf,",unorm);
  for(i=0;i<u->np;i++)
   {
    if(i>0)fprintf(fid,",");
    fprintf(fid,"%lf",(u->p)[i]);
   }
  fprintf(fid,")");

  return;
 }
