/* 
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *      author: Mike Henderson mhender@watson.ibm.com
 *      date:   November 11, 1997
 *              February 2, 1999   Ported to C
 */

static char *id="@(#) $Id: MFKVector.c 257 2006-08-30 20:30:07Z mhender $";

#include <MFErrorHandler.h>
#include <MFKVector.h>
#include <float.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#ifndef DBL_QNAN
#define DBL_QNAN 1.e200
#endif

void MFSetError(MFErrorHandler,int,char*,char*,int,char*);
static char MFKVectorErrorMsg[256]="";

struct MFKVectorSt
 {
  int nC;
  double *data;
  int nRefs;
 };

MFKVector MFCreateKVector(int n, MFErrorHandler e)
 {
  static char RoutineName[]={"MFCreateKVector"};
  MFKVector this;
  int i;

#ifdef MFNOCONFIDENCE
  if(n<1)
   {
    sprintf(MFKVectorErrorMsg,"Length of Vector %d (argument 1) is Illegal. Must be positive.",n);
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  this=malloc(sizeof(struct MFKVectorSt)); /*done*/

#ifdef MFNOSAFETYNET
  if(this==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFKVectorSt));
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif

  this->nC=n;

  this->data=malloc(n*sizeof(double));

#ifdef MFNOSAFETYNET
  if(this->data==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Out of memory, trying to allocate %d bytes",n*sizeof(double));
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif

  for(i=0;i<n;i++)this->data[i]=0.;
  this->nRefs=1;

  return this;
 }

MFKVector MFCreateKVectorWithData(int n,double *vl, MFErrorHandler e)
 {
  static char RoutineName[]={"MFCreateKVectorWithFullData"};
  int i;
  MFKVector this;

  if(n<1)
   {
    sprintf(MFKVectorErrorMsg,"Length of Vector %d (argument 1) is Illegal. Must be positive.",n);
    MFSetError(e,4,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return NULL;
   }

  this=malloc(sizeof(struct MFKVectorSt)); /*done*/

#ifdef MFNOSAFETYNET
  if(this==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFKVectorSt));
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return this;
   }
#endif

  this->data=malloc(n*sizeof(double)); /*done*/

#ifdef MFNOSAFETYNET
  if(this->data==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Out of memory, trying to allocate %d bytes",n*sizeof(double));
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return this;
   }
#endif

#ifdef MFNOCONFIDENCE
  if(vl==NULL)
   {
    sprintf(MFKVectorErrorMsg,"The pointer to the array of coordinates (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    this->data=NULL;
    this->nRefs=1;
    return this;
   }
#endif

  for(i=0;i<n;i++)this->data[i]=vl[i];
  this->nC=n;

  this->nRefs=1;

  return this;
 }

void MFFreeKVector(MFKVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"MFFreeKVector"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Pointer to Vector (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->nRefs--;

  if(this->nRefs<1)
   {
    if(this->data!=NULL){free(this->data);this->data=NULL;}
    free(this);
   }
  return;
 }

int MFKV_NC(MFKVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"MFKV_NC"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Pointer to Vector (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return -1;
   }
#endif

  return this->nC;
 }

double MFKV_C(MFKVector this,int i, MFErrorHandler e)
 {
  static char RoutineName[]={"MFKV_C"};
  int j;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Pointer to Vector (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return DBL_QNAN;
   }
  if(i<0|| !(i<this->nC))
   {
    sprintf(MFKVectorErrorMsg,"Coordinate %d (argument 2) is illegal. Must be in 0 to %d",i,this->nC-1);
    MFSetError(e,8,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return DBL_QNAN;
   }
#endif

  return this->data[i];
 }

void MFKVSetC(MFKVector this,int i,double vl, MFErrorHandler e)
 {
  static char RoutineName[]={"MFKVSetC"};
  int j;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Pointer to Vector (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(i<0|| !(i<this->nC))
   {
    sprintf(MFKVectorErrorMsg,"Coordinate %d (argument 2) is illegal. Must be in 0 to %d",i,this->nC-1);
    MFSetError(e,8,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->data[i]=vl;
  return;
 }

void MFRefKVector(MFKVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"MFRefKVector"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Pointer to Vector (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->nRefs++;
  return;
 }

double MFKVDot(MFKVector a,MFKVector b, MFErrorHandler e)
 {
  static char RoutineName[]={"MFKVDot"};
  double d;
  int i;

#ifdef MFNOCONFIDENCE
  if(a==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Vector a (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return DBL_QNAN;
   }

  if(b==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Vector b (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return DBL_QNAN;
   }

  if(a->nC!=b->nC)
   {
    sprintf(MFKVectorErrorMsg,"Vectors must be the same length a=%d, b=%d",a->nC,b->nC);
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return DBL_QNAN;
   }
#endif

  d=0;
  for(i=0;i<a->nC;i++)d+=a->data[i]*b->data[i];

  return d;
 }

double MFKVNorm(MFKVector a, MFErrorHandler e)
 {
  static char RoutineName[]={"MFKVNorm"};
  double d;
  int i;

#ifdef MFNOCONFIDENCE
  if(a==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Vector a (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return -1.;
   }
#endif

  d=0;
  for(i=0;i<a->nC;i++)d+=a->data[i]*a->data[i];

  return sqrt(d);
 }

void MFKVScale(double s,MFKVector a, MFErrorHandler e)
 {
  static char RoutineName[]={"MFKVScale"};
  int i;

  for(i=0;i<a->nC;i++)a->data[i]*=s;

  return;
 }

void MFKVScaleMul(double s,MFKVector a,MFKVector b, MFErrorHandler e)
 {
  static char RoutineName[]={"MFKVScaleMul"};
  int i;

#ifdef MFNOCONFIDENCE
  if(a==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Vector a (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(b==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Vector a (argument 3) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(a->nC!=b->nC)
   {
    sprintf(MFKVectorErrorMsg,"Vectors must be the same length a=%d, b=%d",a->nC,b->nC);
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  for(i=0;i<a->nC;i++)b->data[i]=s*a->data[i];

  return;
 }

double *MFKV_CStar(MFKVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"MFKV_CStar"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Pointer to Vector (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  return this->data;
 }

void MFKVDiff(MFKVector a,MFKVector b, MFKVector c, MFErrorHandler e)
 {
  static char RoutineName[]={"MFKVDiff"};
  MFKVector result;
  int i,n;

#ifdef MFNOCONFIDENCE
  if(a==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Vector a (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(b==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Vector b (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(c==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Vector c (argument 3) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(a->nC!=b->nC || a->nC!=c->nC || b->nC!=c->nC)
   {
    sprintf(MFKVectorErrorMsg,"Vectors must all be the same length a=%d, b=%d, c=%d",a->nC,b->nC,c->nC);
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  n=a->nC;

  for(i=0;i<n;i++)
    c->data[i]=a->data[i]-b->data[i];

  return;
 }

void MFKVAdd(MFKVector a,MFKVector b, MFKVector c, MFErrorHandler e)
 {
  static char RoutineName[]={"MFKVAdd"};
  MFKVector result;
  int i,n;

#ifdef MFNOCONFIDENCE
  if(a==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Vector a (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(b==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Vector b (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(c==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Vector c (argument 3) is NULL");
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(a->nC!=b->nC || a->nC!=c->nC || b->nC!=c->nC)
   {
    sprintf(MFKVectorErrorMsg,"Vectors must all be the same length a=%d, b=%d, c=%d",a->nC,b->nC,c->nC);
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  n=a->nC;

  for(i=0;i<n;i++)
    c->data[i]=a->data[i]+b->data[i];

  return;
 }

void MFWriteKVector(FILE *fid,MFKVector s, MFErrorHandler e)
 {
  static char RoutineName[]={"MFWriteKVector"};
  int i;

  fprintf(fid,"%s\n","KVector");
  fprintf(fid,"%d %d\n",s->nC,s->nRefs);
  for(i=0;i<s->nC;i++)
   {
    if(i>0)fprintf(fid," ");
    fprintf(fid,"%lf",s->data[i]);
   }
  fprintf(fid,"\n");

  return;
 }

MFKVector MFReadKVector(FILE *fid, MFErrorHandler e)
 {
  static char RoutineName[]={"MFReadKVector"};
  int i;
  MFKVector s;
  char tag[100]="";

  fscanf(fid,"%s\n",tag);

#ifdef MFNOCONFIDENCE
  if(strcmp(tag,"KVector"))
   {
    sprintf(MFKVectorErrorMsg,"Next Object is not a KVector! (%s)\n",RoutineName,tag);
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  s=malloc(sizeof(struct MFKVectorSt)); /*done*/
#ifdef MFNOSAFETYNET
  if(s==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFKVectorSt));
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif

  fscanf(fid,"%d %d\n",&(s->nC),&(s->nRefs));

  s->data=malloc(s->nC*sizeof(double)); /*done*/

#ifdef MFNOSAFETYNET
  if(s->data==NULL)
   {
    sprintf(MFKVectorErrorMsg,"Out of memory, trying to allocate %d bytes",s->nC*sizeof(double));
    MFSetError(e,12,RoutineName,MFKVectorErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif
  
  for(i=0;i<s->nC;i++)
   {
    if(i>0)fscanf(fid," ");
    fscanf(fid,"%lf",&(s->data[i]));
   }
  fscanf(fid,"\n");

  return s;
 }

int MFKVectorGetNRefs(MFKVector u, MFErrorHandler e)
 {
  return u->nRefs;
 }
