/*
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *      author: Mike Henderson mhender@watson.ibm.com
 *      date:   April 1, 2002
 */

static char *id="@(#) $Id: MFHenderson.c 307 2006-09-27 20:57:43Z mhender $";

static char MFHendersonErrorMsg[256];

#include <MFAtlas.h>
#include <MFContinuationMethod.h>
#include <MFImplicitMF.h>
#include <MFNRegion.h>
#include <MFNVector.h>
#include <MFPrint.h>
#include <stdio.h>

static FILE *restartReg=NULL;
static FILE *restartSing=NULL;
static double *x=NULL;

void MFAtlasPageOutChartsNotNearBoundary(MFAtlas,int,int,char*,MFErrorHandler);
void MFAtlasWriteOutChartsNotPaged(MFAtlas,int,int,char*,MFErrorHandler);
int MFNVectorGetNRefs(MFNVector,MFErrorHandler);
MFChart MFAtlasChart(MFAtlas,int,MFErrorHandler);

void MFAtlasPageOutAllCharts(MFAtlas,MFErrorHandler);

static void MFHendersonExtendAtlas(MFContinuationMethod H,MFAtlas A, MFImplicitMF M, MFNRegion Omega, int m, MFNVector *u0, MFNKMatrix *Phi,MFErrorHandler);
static void MFHendersonMethodCopyClipF(MFContinuationMethod,MFAtlas,MFErrorHandler);
static void MFHendersonCloseAtlas(MFContinuationMethod,MFAtlas,MFErrorHandler);
static void MFHendersonFlushAtlas(MFContinuationMethod,MFAtlas,MFErrorHandler);
static void MFHendersonFreeParmBlock(void*,MFErrorHandler);

struct MFHendersonParmBlockST {
                             int verbose;
                             int maxCharts;
                             double minR;
                             double maxR;
                             double epsilon;
                             double dotmin;
                             int page;
                             int pageEvery;
                             int useBB;
                             int dumpToPlotFile;
                             int dumpToCenterFile;
                             int dumpToRestartFile;
                             int dumpToRestartFileEvery;
                             char *fileName;
                             int checkPoint;
                             int checkPointEvery;
                             int branchSwitch;
                             int nClipF;
                             int mClipF;
                             double (**clipF)(MFNVector,MFErrorHandler);
                            };
typedef struct MFHendersonParmBlockST *MFHendersonParmBlock;

MFContinuationMethod MFCreateHendersonsMethod(MFErrorHandler e)
 {
  static char RoutineName[]={"MFCreateHendersonsMethod"};
  MFContinuationMethod result;
  MFHendersonParmBlock parms;

  parms=malloc(sizeof(struct MFHendersonParmBlockST));

#ifndef MFNOSAFETYNET
  if(parms==NULL)
   {
    sprintf(MFHendersonErrorMsg,"Out of space trying to allocate %d bytes.",sizeof(struct MFHendersonParmBlockST));
    MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    MFFreeContinuationMethod(result,e);
    return NULL;
   }
#endif

  parms->verbose=0;
  parms->maxCharts=-1;
  parms->minR=.01;
  parms->maxR=1.;
  parms->epsilon=1.e-7;
  parms->dotmin=.2;
  parms->page=1;
  parms->pageEvery=1000;
  parms->useBB=1;
  parms->dumpToPlotFile=1;
  parms->dumpToCenterFile=1;
  parms->dumpToRestartFile=1;
  parms->dumpToRestartFileEvery=100;
  parms->checkPoint=0;
  parms->checkPointEvery=100;
  parms->branchSwitch=1;
  parms->fileName=malloc(6*sizeof(char));

#ifndef MFNOSAFETYNET
  if(parms->fileName==NULL)
   {
    sprintf(MFHendersonErrorMsg,"Out of space trying to allocate %d bytes.",6*sizeof(char));
    MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    MFFreeContinuationMethod(result,e);
    return NULL;
   }
#endif

  strcpy(parms->fileName,"Atlas");

  parms->nClipF=0;
  parms->mClipF=0;
  parms->clipF=NULL;

  result=MFCreateContinuationMethodBaseClass("Henderson",e);
  MFContinuationMethodSetParmBlock(result,parms,e);
  MFContinuationMethodSetFreeParmBlock(result,MFHendersonFreeParmBlock,e);
  MFContinuationMethodSetCloseAtlas(result,MFHendersonCloseAtlas,e);
  MFContinuationMethodSetExtendAtlasMultipleWithTangents(result,MFHendersonExtendAtlas,e);
  MFContinuationMethodSetFlushAtlas(result,MFHendersonFlushAtlas,e);

  return result;
 }

void MFHendersonFreeParmBlock(void *parmBlock, MFErrorHandler e)
 { 
  MFHendersonParmBlock parms;

  parms=(MFHendersonParmBlock)(parmBlock);
  if(parms->fileName!=NULL)free(parms->fileName);
  if(parms->clipF!=NULL)free(parms->clipF);
  free(parms);

  return;
 }

char *MFHendersonGetFilename(MFContinuationMethod H, MFErrorHandler e)
 {
  static char RoutineName[]={"MFHendersonGetFileName"};
  MFHendersonParmBlock parms;

#ifdef MFNOCONFIDENCE
  if(H==NULL)
   {
    sprintf(MFHendersonErrorMsg,"MFContinuation Method (argument 1) is NULL.");
    MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(strcmp(MFContinuationMethodGetType(H,e),"Henderson"))
   {
    sprintf(MFHendersonErrorMsg,"MFContinuation Method (argument 1) is not Henderson's Method.");
    MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  parms=(MFHendersonParmBlock)(MFContinuationMethodGetParmBlock(H,e));
  return parms->fileName;
 }

void MFHendersonSetFilename(MFContinuationMethod H,char *name, MFErrorHandler e)
 {
  static char RoutineName[]={"MFHendersonSetFileName"};
  MFHendersonParmBlock parms;

#ifdef MFNOCONFIDENCE
  if(H==NULL)
   {
    sprintf(MFHendersonErrorMsg,"MFContinuation Method (argument 1) is NULL.");
    MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(strcmp(MFContinuationMethodGetType(H,e),"Henderson"))
   {
    sprintf(MFHendersonErrorMsg,"MFContinuation Method (argument 1) is not Henderson's Method.");
    MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  parms=(MFHendersonParmBlock)(MFContinuationMethodGetParmBlock(H,e));
  free(parms->fileName);
  parms->fileName=malloc((strlen(name)+1)*sizeof(char));

#ifndef MFNOSAFETYNET
  if(parms->fileName==NULL)
   {
    sprintf(MFHendersonErrorMsg,"Out of space trying to allocate %d bytes.",(strlen(name)+1)*sizeof(char));
    MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return;
   }
#endif

  strcpy(parms->fileName,name);
  return;
 }

void MFHendersonAddClipF(MFContinuationMethod H,double (*clipF)(MFNVector,MFErrorHandler), MFErrorHandler e)
 {
  static char RoutineName[]={"MFHendersonAddClipF"};
  MFHendersonParmBlock this;
  int verbose=0;

#ifdef MFALLOWVERBOSE
  if(verbose){printf("%s\n",RoutineName);fflush(stdout);}
#endif

  this=(MFHendersonParmBlock)(MFContinuationMethodGetParmBlock(H,e));

  if(this->nClipF>=this->mClipF)
   {
    this->mClipF+=10;
    this->clipF=realloc(this->clipF,(this->mClipF)*sizeof(double (*)(MFNVector,MFErrorHandler)));

#ifndef MFNOSAFETYNET
  if(this->clipF==NULL)
   {
    sprintf(MFHendersonErrorMsg,"Out of space trying to allocate %d bytes.",(this->mClipF)*sizeof(double (*)(MFNVector)));
    MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return;
   }
#endif

   }
  this->clipF[this->nClipF]=clipF;
  this->nClipF++;
 }

void MFHendersonMethodCopyClipF(MFContinuationMethod H,MFAtlas A, MFErrorHandler e)
 {
  static char RoutineName[]={"MFHendersonsMethodCopyClipF"};
  int i;
  MFHendersonParmBlock this;
  int verbose=0;

#ifdef MFALLOWVERBOSE
  if(verbose){printf("%s\n",RoutineName);fflush(stdout);}
#endif

  this=(MFHendersonParmBlock)(MFContinuationMethodGetParmBlock(H,e));

  MFAtlasClearClipF(A,e);
  for(i=0;i<this->nClipF;i++)MFAtlasAddClipF(A,this->clipF[i],e);

  return;
 }

void MFHendersonsMethodClearClipF(MFContinuationMethod H, MFErrorHandler e)
 {
  static char RoutineName[]={"MFHendersonsMethodClearClipF"};
  MFHendersonParmBlock this;
  int i;
  int verbose=0;

#ifdef MFALLOWVERBOSE
  if(verbose){printf("%s\n",RoutineName);fflush(stdout);}
#endif

  this=(MFHendersonParmBlock)(MFContinuationMethodGetParmBlock(H,e));

  for(i=0;i<this->nClipF;i++)this->clipF[i]=NULL;

  this->nClipF=0;

  return;
 }

void MFHendersonExtendAtlas(MFContinuationMethod H, MFAtlas S, MFImplicitMF M, MFNRegion Omega, int m, MFNVector *u0, MFNKMatrix *Phi0, MFErrorHandler e)
 {
  static char RoutineName[]={"MFHendersonExtendAtlas"};
  int i,j;
  int kkk;
  double delta=0.;
  int chart;
  int n,N;
  MFNVector u;
  MFNKMatrix Phi;
  double R,Rs;
  FILE *fid;
  double volume;
  double epsilon;
  double dotmin;
  int kmax;
  int page,pageEvery;
  int checkPoint,checkPointEvery;
  int branchSwitch;
  int dumpToPlotFile;
  int dumpToCenterFile;
  int rc;
  int verbose;
  int switched;
  int dumpToRestartFile;
  int dumpToRestartFileEvery;
  char *name;
  char FileName[4096];

/* If M!=AtlasM change it. What if operations between n vectors don't work? */

  if(m==0)return;

  epsilon=MFHendersonGetRealParameter(H,"epsilon",e);
  dotmin=MFHendersonGetRealParameter(H,"dotmin",e);
  kmax=MFHendersonGetIntegerParameter(H,"maxCharts",e);
  verbose=MFHendersonGetIntegerParameter(H,"verbose",e);
  page=MFHendersonGetIntegerParameter(H,"page",e);
  pageEvery=MFHendersonGetIntegerParameter(H,"pageEvery",e);
  checkPoint=MFHendersonGetIntegerParameter(H,"checkPoint",e);
  checkPointEvery=MFHendersonGetIntegerParameter(H,"checkPointEvery",e);
  branchSwitch=MFHendersonGetIntegerParameter(H,"branchSwitch",e);
  dumpToPlotFile=MFHendersonGetIntegerParameter(H,"dumpToPlotFile",e);
  dumpToCenterFile=MFHendersonGetIntegerParameter(H,"dumpToCenterFile",e);
  name=MFHendersonGetFilename(H,e);
  switched=0;
  MFHendersonMethodCopyClipF(H,S,e);
  dumpToRestartFile=MFHendersonGetIntegerParameter(H,"dumpToRestartFile",e);
  dumpToRestartFileEvery=MFHendersonGetIntegerParameter(H,"dumpToRestartFileEvery",e);

#ifdef MFALLOWVERBOSE
  if(verbose){printf("%s, epsilon=%lf, kmax=%d, verbose=%d\n",RoutineName,epsilon,kmax,verbose);fflush(stdout);}
#endif

  kkk=0;
  MFAtlasSetEpsilon(S,epsilon,e);
  MFAtlasSetDotMin(S,dotmin,e);
  MFAtlasSetVerbose(S,verbose,e);
  for(i=0;i<m;i++)
   {
    if(Phi0==NULL||Phi0[i]==NULL)
      rc=MFAtlasAddChart(S,u0[i],e);
     else{
      R=MFIMFScale(M,u0[i],Phi0[i],e);
      rc=MFAtlasAddChartWithAll(S,u0[i],Phi0[i],R,e);
     }

#ifdef MFALLOWVERBOSE
    if(verbose){printf("%d) Add initial points to Atlas ",rc);MFPrintNVector(stdout,u0[i],e);printf(" index %d, bif? %d\n",MFNVGetIndex(u0[i],e),MFNVGetIndex2(u0[i],e));fflush(stdout);}
#endif

    MFChartSetReferenceNumber(MFAtlasChart(S,rc,e),0,e);
#ifdef MFVOLUME
    volume=MFVolumeOfAtlas(S,Omega,e);

#ifdef MFALLOWVERBOSE
    if(verbose){printf("  Volume of Atlas %lf\n",volume);fflush(stdout);}
#endif

#endif
    kkk++;

    if(MFChartReferenceNumber(MFAtlasChart(S,rc,e),e)>dumpToRestartFileEvery)
     {
      printf("Reference Number has exceeded the threshold\n");fflush(stdout);
      MFChartSetReferenceNumber(MFAtlasChart(S,rc,e),0,e);
      MFUpdateNeighborsReferenceMarks(S,rc,dumpToRestartFileEvery,e);
     }
    if(dumpToRestartFile && MFChartReferenceNumber(MFAtlasChart(S,rc,e),e)==0)
     {
      if(MFChartIsSingular(MFAtlasChart(S,rc,e),e))
       {
        if(restartSing==NULL)
         {
          strcpy(FileName,name);
          strcat(FileName,".singular");
          restartSing=fopen(FileName,"w");
          if(restartSing==NULL)abort();
         }
        N=MFIMFProjectToDraw(MFAtlasMF(S,e),NULL,NULL,e);
        if(x==NULL)
         {
          x=malloc(N*sizeof(double));

#ifndef MFNOSAFETYNET
          if(x==NULL)
           {
            sprintf(MFHendersonErrorMsg,"Out of space trying to allocate %d bytes.",N*sizeof(double));
            MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
            MFErrorHandlerOutOfMemory(e);
            return;
           }
#endif
         }

        MFIMFProjectToDraw(MFAtlasMF(S,e),MFChartCenter(MFAtlasChart(S,rc,e),e),x,e);
        for(j=0;j<N;j++)fprintf(restartSing," %lf",x[j]);
        fprintf(restartSing,"\n");
       }else{
        if(restartReg==NULL)
         {
          strcpy(FileName,name);
          strcat(FileName,".regular");
          restartReg=fopen(FileName,"w");
          if(restartReg==NULL)abort();
         }
        N=MFIMFProjectToDraw(MFAtlasMF(S,e),NULL,NULL,e);
        if(x==NULL)
         {
          x=malloc(N*sizeof(double));

#ifndef MFNOSAFETYNET
          if(x==NULL)
           {
            sprintf(MFHendersonErrorMsg,"Out of space trying to allocate %d bytes.",N*sizeof(double));
            MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
            MFErrorHandlerOutOfMemory(e);
            return;
           }
#endif

         }
        MFIMFProjectToDraw(MFAtlasMF(S,e),MFChartCenter(MFAtlasChart(S,rc,e),e),x,e);
        for(j=0;j<N;j++)fprintf(restartReg," %lf",x[j]);
        fprintf(restartReg,"\n");
       }
     }

    if(page && kkk%pageEvery==0)
     {
      MFAtlasPageOutChartsNotNearBoundary(S,dumpToPlotFile,dumpToCenterFile,name,e);
     }
    if(checkPoint && kkk%checkPointEvery==0)
     {

#ifdef MFALLOWVERBOSE
      if(verbose){printf("Check pointing, step %d\n",kkk);fflush(stdout);}
#endif

      fid=fopen("Checkpoint.atlas","w");
      MFWriteAtlas(fid,S,e);
      fclose(fid);
     }
   }

  n=MFAtlasN(S,e);
  u=MFCloneNVector(u0[0],e);

GoOn:
  rc=1;
  while(rc>0&&(kmax<0||kkk<kmax)&&(chart=MFAtlasPointOnBoundaryInsideRegion(S,Omega,u,&Phi,&delta,e))>-1)
   {
    R=MFIMFScale(MFAtlasMF(S,e),u,Phi,e);
    Rs=2*MFAtlasChartSuggestedRadius(S,chart,e);
    if(0&&R>Rs)R=Rs;

    if((rc=MFAtlasAddChartWithAll(S,u,Phi,R,e))>0)
     {

#ifdef MFALLOWVERBOSE
      if(verbose){printf("%d) Add boundary point to Atlas ",rc);MFPrintNVector(stdout,u,e);printf(" index %d, bif? %d\n",MFNVGetIndex(u,e),MFNVGetIndex2(u,e));fflush(stdout);}
#endif

      if(MFChartReferenceNumber(MFAtlasChart(S,rc,e),e)>dumpToRestartFileEvery)
       {
        printf("Reference Number has exceeded the threshold\n");fflush(stdout);
        MFChartSetReferenceNumber(MFAtlasChart(S,rc,e),0,e);
        MFUpdateNeighborsReferenceMarks(S,rc,dumpToRestartFileEvery,e);
       }
/*    printf(" Reference Number is %d\n",MFChartReferenceNumber(MFAtlasChart(S,rc,e),e));*/
      if(dumpToRestartFile && MFChartReferenceNumber(MFAtlasChart(S,rc,e),e)==0)
       {
        if(MFChartIsSingular(MFAtlasChart(S,rc,e),e))
         {
          if(restartSing==NULL)
           {
            strcpy(FileName,name);
            strcat(FileName,".singular");
            restartSing=fopen(FileName,"w");
            if(restartSing==NULL)abort();
           }
          N=MFIMFProjectToDraw(MFAtlasMF(S,e),NULL,NULL,e);
          if(x==NULL)
           {
            x=malloc(N*sizeof(double));

#ifndef MFNOSAFETYNET
            if(x==NULL)
             {
              sprintf(MFHendersonErrorMsg,"Out of space trying to allocate %d bytes.",N*sizeof(double));
              MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
              MFErrorHandlerOutOfMemory(e);
              return;
             }
#endif
           }

          MFIMFProjectToDraw(MFAtlasMF(S,e),MFChartCenter(MFAtlasChart(S,rc,e),e),x,e);
          for(j=0;j<N;j++)fprintf(restartSing," %lf",x[j]);
          fprintf(restartSing,"\n");
         }else{
          if(restartReg==NULL)
           {
            strcpy(FileName,name);
            strcat(FileName,".regular");
            restartReg=fopen(FileName,"w");
            if(restartReg==NULL)abort();
           }
          N=MFIMFProjectToDraw(MFAtlasMF(S,e),NULL,NULL,e);
          if(x==NULL)
           {
            x=malloc(N*sizeof(double));

#ifndef MFNOSAFETYNET
            if(x==NULL)
             {
              sprintf(MFHendersonErrorMsg,"Out of space trying to allocate %d bytes.",N*sizeof(double));
              MFSetError(e,12,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
              MFErrorHandlerOutOfMemory(e);
              return;
             }
#endif
           }
          MFIMFProjectToDraw(MFAtlasMF(S,e),MFChartCenter(MFAtlasChart(S,rc,e),e),x,e);
          for(j=0;j<N;j++)fprintf(restartReg," %lf",x[j]);
          fprintf(restartReg,"\n");
         }
       }
#ifdef MFVOLUME
      volume=MFVolumeOfAtlas(S,Omega);

#ifdef MFALLOWVERBOSE
      if(verbose){printf("  Volume of Atlas %lf\n",volume);fflush(stdout);}
#endif

#endif

      kkk++;
      if(page && kkk%pageEvery==0)
       {
        MFAtlasPageOutChartsNotNearBoundary(S,dumpToPlotFile,dumpToCenterFile,name,e);
       }
      if(checkPoint && kkk%checkPointEvery==0)
       {

#ifdef MFALLOWVERBOSE
        if(verbose){printf("Check pointing, step %d\n",kkk);fflush(stdout);}
#endif

        fid=fopen("Checkpoint.atlas","w");
        MFWriteAtlas(fid,S,e);
        fclose(fid);
       }
     }

    MFFreeNVector(u,e);
    MFFreeNKMatrix(Phi,e);
    u=MFCloneNVector(u0[0],e);
   }

#ifdef MFALLOWVERBOSE
  if(verbose&&chart<0){printf("No more points on the boundary, switch=%d, switched=%d\n",branchSwitch,switched);fflush(stdout);}
   else if(verbose){printf("Maximum number of charts added (%d)\n",kkk);fflush(stdout);}
#endif

  if((switched<branchSwitch) && branchSwitch!=0 && chart<0)
   {
    int schart,bchart;
    MFNVector u1,u;
    MFNKMatrix Phi;

    printf("Branch Switching (using one of the singular charts to get a new boundary point\n");fflush(stdout);

    schart=MFAtlasGetSingularChartWithBoundary(S,Omega,e);
    if(schart>0)
     {

#ifdef MFALLOWVERBOSE
      if(1||verbose){printf("Branch switching, singular chart with boundary is %d, index %d, Radius %lf\n",schart,MFNVGetIndex2(MFAtlasChartCenter(S,schart,e),e),MFAtlasChartRadius(S,schart,e));fflush(stdout);}
#endif

/* This is the branch that passes on through. */

      u1=MFAtlasGetPointOnBoundaryChart(S,Omega,schart,1.,e);
      if(u1==NULL)goto GoOn;

      Phi=MFAtlasChartTangentSpace(S,schart,e);
      R=MFIMFScale(MFAtlasMF(S,e),u1,Phi,e);
      MFIMFSetStability(MFAtlasMF(S,e),u1,Phi,e);
      bchart=MFAtlasAddChartWithAll(S,u1,Phi,R,e);

#ifdef MFALLOWVERBOSE
      if(1||verbose){printf("%d) Add bifurcating chart to Atlas ",bchart);MFPrintNVector(stdout,u1,e);printf(" index %d, bif? %d\n",MFNVGetIndex(u1,e),MFNVGetIndex2(u1,e));fflush(stdout);}
#endif

      MFFreeNKMatrix(Phi,e);

      switched+=1;
      goto GoOn;
     }
   }

  MFFreeNVector(u,e);

  return;
 }

void MFHendersonCloseAtlas(MFContinuationMethod H, MFAtlas S, MFErrorHandler e)
 {
  static char RoutineName[]={"MFCloseAtlas"};
  int page;
  int dumpToPlotFile;
  int dumpToCenterFile;
  char *name;

  page=MFHendersonGetIntegerParameter(H,"page",e);
  dumpToPlotFile=MFHendersonGetIntegerParameter(H,"dumpToPlotFile",e);
  dumpToCenterFile=MFHendersonGetIntegerParameter(H,"dumpToCenterFile",e);
  name=MFHendersonGetFilename(H,e);

  if(dumpToPlotFile)MFAtlasWriteOutChartsNotPaged(S,dumpToPlotFile,dumpToCenterFile,name,e);
  if(restartSing!=NULL){fclose(restartSing);restartSing=NULL;}
  if(restartReg!=NULL){fclose(restartReg);restartReg=NULL;}
  if(x!=NULL){free(x);x=NULL;}

  MFAtlasClosePlotfile(S,e);
  MFAtlasCloseCenterfile(S,e);

  return;
 }

void MFHendersonFlushAtlas(MFContinuationMethod H, MFAtlas S, MFErrorHandler e)
 {
  static char RoutineName[]={"MFFlushAtlas"};
  int page;
  int dumpToPlotFile;
  int dumpToCenterFile;
  char *name;

  page=MFHendersonGetRealParameter(H,"page",e);
  dumpToPlotFile=MFHendersonGetRealParameter(H,"dumpToPlotFile",e);
  dumpToCenterFile=MFHendersonGetIntegerParameter(H,"dumpToCenterFile",e);
  name=MFHendersonGetFilename(H,e);

  if(page)MFAtlasWriteOutChartsNotPaged(S,dumpToPlotFile,dumpToCenterFile,name,e);

  return;
 }

/*!    \fn int MFHendersonSetIntegerParameter(MFImplicitMF M, char *parameterName, int value)
 *     \brief Allows the user to set integer parameters.
 *
 * Legal integer parameter names and their default values.
 *  <ul>
 *    <li>                   verbose                      default:    0
 *    <li>                   maxCharts                    default:    1
 *    <li>                   page                         default:    1
 *    <li>                   pageEvery                    default: 1000
 *    <li>                   useBB                        default:    1
 *    <li>                   dumpToPlotFile               default:    1
 *    <li>                   dumpToCenterFile             default:    1
 *    <li>                   dumpToRestartFile            default:    1
 *    <li>                   dumpToRestartFileEvery       default:  100
 *    <li>                   checkPoint                   default:    0
 *    <li>                   checkPointEvery              default:  100
 *    <li>                   branchSwitch                 default:    1
 *  </ul>
 *     \param M An Henderson solution manifold.
 *     \param parameterName Which parameter to set.
 *     \param value The new value.
 *     \returns FALSE if the parameter name does not match a parameter.
 */
int MFHendersonSetIntegerParameter(MFContinuationMethod M, char *parameterName, int value, MFErrorHandler e)
 {
  static char RoutineName[]={"MFHendersonSetIntegerParameter"};
  struct MFHendersonParmBlockST *data;

  if(strcmp(MFContinuationMethodGetType(M,e),"Henderson"))
   {
    sprintf(MFHendersonErrorMsg,"Parameter 1 must be a Hendersons algorithm");
    MFSetError(e,4,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    return 0;
   }

  data=(struct MFHendersonParmBlockST*)MFContinuationMethodGetParmBlock(M,e);

/*
 *if(!strcmp(parameterName,"verbose"))printf("set verbose to %d\n",value);
 * else if(!strcmp(parameterName,"maxCharts"))printf("set maxCharts to %d\n",value);
 * else if(!strcmp(parameterName,"page"))printf("set maxCharts to %d\n",value);
 * else if(!strcmp(parameterName,"pageEvery"))printf("set pageEvery to %d\n",value);
 * else if(!strcmp(parameterName,"useBB"))printf("set useBB to %d\n",value);
 * else if(!strcmp(parameterName,"dumpToPlotFile"))printf("set dumpToPlotfile to %d\n",value);
 * else if(!strcmp(parameterName,"dumpToCenterFile"))printf("set dumpToCenterFile to %d\n",value);
 * else if(!strcmp(parameterName,"dumpToRestartFile"))printf("set dumpToRestartFile to %d\n",value);
 * else if(!strcmp(parameterName,"dumpToRestartFileEvery"))printf("set dumpToRestartFileEvery to %d\n",value);
 * else if(!strcmp(parameterName,"checkPoint"))printf("set checkPoints to %d\n",value);
 * else if(!strcmp(parameterName,"checkPointEvery"))printf("set checkPointEvery to %d\n",value);
 * else if(!strcmp(parameterName,"branchSwitch"))printf("set branchSwitch to %d\n",value);
 * else printf("unknown parameter name \"%s\"\n",parameterName);
 *fflush(stdout);
 */

  if(!strcmp(parameterName,"verbose"))data->verbose=value;
   else if(!strcmp(parameterName,"maxCharts"))data->maxCharts=value;
   else if(!strcmp(parameterName,"page"))data->page=value;
   else if(!strcmp(parameterName,"pageEvery"))data->pageEvery=value;
   else if(!strcmp(parameterName,"useBB"))data->useBB=value;
   else if(!strcmp(parameterName,"dumpToPlotFile"))data->dumpToPlotFile=value;
   else if(!strcmp(parameterName,"dumpToCenterFile"))data->dumpToCenterFile=value;
   else if(!strcmp(parameterName,"dumpToRestartFile"))data->dumpToRestartFile=value;
   else if(!strcmp(parameterName,"dumpToRestartFileEvery"))data->dumpToRestartFileEvery=value;
   else if(!strcmp(parameterName,"checkPoint"))data->checkPoint=value;
   else if(!strcmp(parameterName,"checkPointEvery"))data->checkPointEvery=value;
   else if(!strcmp(parameterName,"branchSwitch"))data->branchSwitch=value;
   else return 0;

  return 1;
 }

/*!    \fn int MFHendersonSetRealParameter(MFContinuationMethod M, char *parameterName, double value)
 *     \brief Allows the user to set real valued parameters.
 *
 * Legal real parameter names and their default values.
 *   <ul>
 *     <li>                   minR                        default:    0.01
 *     <li>                   maxR                        default:    1.
 *     <li>                   epsilon                     default:    1.e-7
 *     <li>                   dotmin                      default:    0.2
 *   </ul>
 *    \param M An Henderson solution manifold.
 *    \param parameterName Which parameter to set.
 *    \param value The new value.
 *    \returns FALSE if the parameter name does not match a parameter.
 */
int MFHendersonSetRealParameter(MFContinuationMethod M, char *parameterName, double value, MFErrorHandler e)
 {
  static char RoutineName[]={"MFHendersonSetRealParameter"};
  struct MFHendersonParmBlockST *data;

  if(strcmp(MFContinuationMethodGetType(M,e),"Henderson"))
   {
    sprintf(MFHendersonErrorMsg,"Parameter 1 must be an Henderson");
    MFSetError(e,4,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    return 0;
   }

  data=(struct MFHendersonParmBlockST*)MFContinuationMethodGetParmBlock(M,e);

  if(!strcmp(parameterName," minR"))data->minR=value;
   else if(!strcmp(parameterName,"maxR"))data->maxR=value;
   else if(!strcmp(parameterName,"epsilon"))data->epsilon=value;
   else if(!strcmp(parameterName,"dotmin"))data->dotmin=value;
   else return 0;

  return 1;
 }

/*!    \fn int MFHendersonGetIntegerParameter(MFContinuationMethod M, char *parameterName)
 *     \brief Allows the user to set integer parameters. These are usually read from a file. Instead, default values
 *            are used when the Henderson is created, and the user may change them.
 *
 * Legal integer parameter names.
 *  <ul>
 *    <li>                   verbose                      1 if output wanted, 0 OW
 *    <li>                   maxCharts                    The maximum number of charts to add.
 *    <li>                   page                         Whether or not to page interior charts.
 *    <li>                   pageEvery                    The interval (number of charts) between checks for paging.
 *    <li>                   useBB                        1 if bounding boxes are to be used.
 *    <li>                   dumpToPlotFile               1 if a plotfile, containing the chart polygons is wanted.
 *    <li>                   dumpToCenterFile             1 is a centerfile, with the centers of the charts, is wanted
 *    <li>                   dumpToRestartFile            1 is a restart file is wanted.
 *    <li>                   dumpToRestartFileEvery       The interval (number of charts) at which a restart file is written.
 *    <li>                   checkPoint                   1 is a checkpoint file is wanted.
 *    <li>                   checkPointEvery              The interval (number of charts) at which a checkpoint file is written.
 *    <li>                   branchSwitch                 1 if the algorithm is to branch switch, 0 OW.
 *  </ul>
 *
 *     \param M An Henderson solution manifold.
 *     \param parameterName Which parameter value to retreive. A warning is issued if the parameter name does not match a parameter.
 *     \returns The current value of the parameter.
 */
int MFHendersonGetIntegerParameter(MFContinuationMethod M, char *parameterName, MFErrorHandler e)
 {
  static char RoutineName[]={"MFHendersonGetIntegerParameter"};
  struct MFHendersonParmBlockST *data;

  if(strcmp(MFContinuationMethodGetType(M,e),"Henderson"))
   {
    sprintf(MFHendersonErrorMsg,"Parameter 1 must be an Henderson");
    MFSetError(e,4,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    return 0;
   }

  data=(struct MFHendersonParmBlockST*)MFContinuationMethodGetParmBlock(M,e);

/*
 *if(!strcmp(parameterName,"verbose"))printf("Getting the verbose parameter value=%d\n",data->verbose);
 * else if(!strcmp(parameterName,"maxCharts"))printf("Getting the maxCharts parameter value=%d\n",data->maxCharts);
 * else if(!strcmp(parameterName,"page"))printf("Getting the page parameter value=%d\n",data->page);
 * else if(!strcmp(parameterName,"pageEvery"))printf("Getting the pageEvery parameter value=%d\n",data->pageEvery);
 * else if(!strcmp(parameterName,"useBB"))printf("Getting the useBB parameter value=%d\n",data->useBB);
 * else if(!strcmp(parameterName,"dumpToPlotFile"))printf("Getting the dumpToPlotFile parameter value=%d\n",data->dumpToPlotFile);
 * else if(!strcmp(parameterName,"dumpToCenterFile"))printf("Getting the dumpToCenterFile parameter value=%d\n",data->dumpToCenterFile);
 * else if(!strcmp(parameterName,"dumpToRestartFile"))printf("Getting the dumpToRestartFile parameter value=%d\n",data->dumpToRestartFile);
 * else if(!strcmp(parameterName,"dumpToRestartFileEvery"))printf("Getting the dumpToRestartFileEvery parameter value=%d\n",data->dumpToRestartFileEvery);
 * else if(!strcmp(parameterName,"checkPoint"))printf("Getting the checkPoint parameter value=%d\n",data->checkPoint);
 * else if(!strcmp(parameterName,"checkPointEvery"))printf("Getting the checkPointEvery parameter value=%d\n",data->checkPointEvery);
 * else if(!strcmp(parameterName,"branchSwitch"))printf("Getting the branchSwitch parameter value=%d\n",data->branchSwitch);
 * else printf("Getting unknown parameter %s\n",parameterName);
 *fflush(stdout);
 */

  if(!strcmp(parameterName,"verbose"))return data->verbose;
   else if(!strcmp(parameterName,"maxCharts"))return data->maxCharts;
   else if(!strcmp(parameterName,"page"))return data->page;
   else if(!strcmp(parameterName,"pageEvery"))return data->pageEvery;
   else if(!strcmp(parameterName,"useBB"))return data->useBB;
   else if(!strcmp(parameterName,"dumpToPlotFile"))return data->dumpToPlotFile;
   else if(!strcmp(parameterName,"dumpToCenterFile"))return data->dumpToCenterFile;
   else if(!strcmp(parameterName,"dumpToRestartFile"))return data->dumpToRestartFile;
   else if(!strcmp(parameterName,"dumpToRestartFileEvery"))return data->dumpToRestartFileEvery;
   else if(!strcmp(parameterName,"checkPoint"))return data->checkPoint;
   else if(!strcmp(parameterName,"checkPointEvery"))return data->checkPointEvery;
   else if(!strcmp(parameterName,"branchSwitch"))return data->branchSwitch;
   else{
    sprintf(MFHendersonErrorMsg,"Parameter %s is not a real valued parameter",parameterName);
    MFSetError(e,4,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    return 0;
   }

  return 0;
 }

/*!    \fn double MFHendersonGetRealParameter(MFContinuationMethod M, char *parameterName)
 *     \brief Allows the user to set real valued parameters. These are usually read from a file. Instead, default values
 *            are used when the Henderson is created, and the user may change them.
 *
 * Legal real parameter names.
 *    <ul>
 *     <li>                   minR      The smallest chart radius to allow before giving up.
 *     <li>                   maxR      The largest chart radius to allow.
 *     <li>                   epsilon   The maximum distance allowed between the tangent space and manifold. (controls stepsize)
 *     <li>                   dotmin    The smallest dot product allowed between adjacent tangent spaces (controls stepsize). A
 *                                              value of 1 means that the tangent spaces must be parallel.
 *    </ul>
 *     \param M An Henderson solution manifold.
 *     \param parameterName Which parameter value to retreive. A warning is issued if the parameter name does not match a parameter.
 *     \returns The current value of the parameter.
 */
double MFHendersonGetRealParameter(MFContinuationMethod M, char *parameterName, MFErrorHandler e)
 {
  static char RoutineName[]={"MFHendersonGetRealParameter"};
  struct MFHendersonParmBlockST *data;

  if(strcmp(MFContinuationMethodGetType(M,e),"Henderson"))
   {
    sprintf(MFHendersonErrorMsg,"Parameter 1 must be an Henderson");
    MFSetError(e,4,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    return 0;
   }

  data=(struct MFHendersonParmBlockST*)MFContinuationMethodGetParmBlock(M,e);

  if(!strcmp(parameterName,"minR"))return data->minR;
   else if(!strcmp(parameterName,"maxR"))return data->maxR;
   else if(!strcmp(parameterName,"epsilon"))return data->epsilon;
   else if(!strcmp(parameterName,"dotmin"))return data->dotmin;
   else{
    sprintf(MFHendersonErrorMsg,"Parameter %s is not a real valued parameter",parameterName);
    MFSetError(e,4,RoutineName,MFHendersonErrorMsg,__LINE__,__FILE__);
    return 0.;
   }

  return 0.;
 }
