/* 
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *
 *      author: Mike Henderson mhender@watson.ibm.com
 *      date:   November 11, 1997
 *              February 2, 1999   Ported to C
 */

static char *id="@(#) $Id: MFDenseNVector.c 257 2006-08-30 20:30:07Z mhender $";

#include <MFErrorHandler.h>
#include <MFNVector.h>
#include <float.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#ifndef DBL_QNAN
#define DBL_QNAN 1.e300
#endif

struct MFDenseNVectorData
 {
  int nC;
  double *data;
  int wrapped;
 };

void MFFreeDenseNVectorData(void*,MFErrorHandler);
int MFDenseNVGetNC(void*,MFErrorHandler);
double MFDenseNVGetC(int,void*,MFErrorHandler);
void MFDenseNVSetC(int,double,void*,MFErrorHandler);
void MFDenseNVDiff(void*,void*,void*,MFErrorHandler);
void MFDenseNVAdd(void*,void*,void*,MFErrorHandler);
void MFWriteDenseNVectorData(FILE*,void*,MFErrorHandler);
MFNVector MFReadDenseNVector(FILE*,MFErrorHandler);
void MFPrintDenseNVector(FILE*,void*,MFErrorHandler);
MFNVector MFCloneDenseNVector(void*,MFErrorHandler);

static char MFNVectorErrorMsg[256]="";

MFNVector MFCreateNVectorWithData(int n,double *vl, MFErrorHandler e)
 {
  static char RoutineName[]={"MFCreateNVectorWithData"};
  struct MFDenseNVectorData *data;
  int i;
  MFNVector this;

#ifdef MFNOCONFIDENCE
  if(n<1)
   {
    sprintf(MFNVectorErrorMsg,"Length of Vector %d (argument 1) is Illegal. Must be positive.",n);
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  data=malloc(sizeof(struct MFDenseNVectorData)); /*done*/

#ifndef MFNOSAFETYNET
  if(data==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFDenseNVectorData));
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return this;
   }
#endif

  data->nC=n;
  data->data=malloc(n*sizeof(double));

#ifndef MFNOSAFETYNET
  if(n>0&&data->data==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Out of memory, trying to allocate %d bytes",n*sizeof(double));
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return this;
   }
#endif

  if(n>0&&vl!=NULL)
   {
    for(i=0;i<n;i++)(data->data)[i]=vl[i];
   }else if(vl==NULL)
   {
    for(i=0;i<n;i++)(data->data)[i]=0.;
   }

  data->wrapped=0;

  this=MFCreateNVectorBaseClass("DENSE",e);

  MFNVectorSetData(this,data,e);
  MFNVectorSetFreeData(this,MFFreeDenseNVectorData,e);
  MFNVectorSetWriteData(this,MFWriteDenseNVectorData,e);
  MFNVectorSetGetNC(this,MFDenseNVGetNC,e);
  MFNVectorSetGetC(this,MFDenseNVGetC,e);
  MFNVectorSetSetC(this,MFDenseNVSetC,e);
  MFNVectorSetDiff(this,MFDenseNVDiff,e);
  MFNVectorSetAdd(this,MFDenseNVAdd,e);
  MFNVectorSetClone(this,MFCloneDenseNVector,e);
  MFNVectorSetPrint(this,MFPrintDenseNVector,e);

  return this;
 }

MFNVector MFCreateWrappedNVector(int n,double *vl, MFErrorHandler e)
 {
  static char RoutineName[]={"MFCreateWrappedNVector"};
  struct MFDenseNVectorData *data;
  int i;
  MFNVector this;

  this=NULL;

#ifdef MFNOCONFIDENCE
  if(n<1)
   {
    sprintf(MFNVectorErrorMsg,"Length of Vector %d (argument 1) is Illegal. Must be positive.",n);
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return NULL;
   }

  if(vl==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to data (argument 2) is Illegal. Must be non-NULL.");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return this;
   }
#endif

  data=malloc(sizeof(struct MFDenseNVectorData)); /*done*/

#ifndef MFNOSAFETYNET
  if(data==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFDenseNVectorData));
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return this;
   }
#endif

  data->nC=n;
  data->data=vl;
  data->wrapped=1;

  this=MFCreateNVectorBaseClass("DENSE",e);

  MFNVectorSetData(this,data,e);
  MFNVectorSetFreeData(this,MFFreeDenseNVectorData,e);
  MFNVectorSetWriteData(this,MFWriteDenseNVectorData,e);
  MFNVectorSetGetNC(this,MFDenseNVGetNC,e);
  MFNVectorSetGetC(this,MFDenseNVGetC,e);
  MFNVectorSetSetC(this,MFDenseNVSetC,e);
  MFNVectorSetDiff(this,MFDenseNVDiff,e);
  MFNVectorSetAdd(this,MFDenseNVAdd,e);
  MFNVectorSetClone(this,MFCloneDenseNVector,e);
  MFNVectorSetPrint(this,MFPrintDenseNVector,e);

  return this;
 }

MFNVector MFCreateNVector(int n, MFErrorHandler e)
 {
  static char RoutineName[]={"MFCreateNVector"};
  int i;
  MFNVector this;

  this=MFCreateNVectorWithData(n,NULL,e);

  return this;
 }

void MFFreeDenseNVectorData(void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"MFFreeDenseNVectorData"};
  struct MFDenseNVectorData *this;

  this=(struct MFDenseNVectorData*)data;

  if(this==NULL)return;
  if(!this->wrapped&&this->data!=NULL)free(this->data);
  free(this);

  return;
 }

int MFDenseNVGetNC(void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"MFNV_NC"};
  struct MFDenseNVectorData *this;

  this=(struct MFDenseNVectorData*)data;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return -1;
   }
#endif

  return this->nC;
 }

double MFDenseNVGetC(int i,void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"MFDenseNVGetC"};
  struct MFDenseNVectorData *this;
  double result;

  this=(struct MFDenseNVectorData*)data;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return DBL_QNAN;
   }

  if(i<0|| !(i<this->nC))
   {
    sprintf(MFNVectorErrorMsg,"Coordinate %d (argument 1) is illegal. Must be in 0 to %d",i,this->nC-1);
    MFSetError(e,8,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return DBL_QNAN;
   }
#endif

  result=this->data[i];

  return result;
 }

void MFDenseNVSetC(int i,double vl,void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"MFDenseNVSetC"};
  struct MFDenseNVectorData *this;

  this=(struct MFDenseNVectorData*)data;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data (argument 3) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(i<0|| !(i<this->nC))
   {
    sprintf(MFNVectorErrorMsg,"Coordinate %d (argument 1) is illegal. Must be in 0 to %d",i,this->nC-1);
    MFSetError(e,8,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->data[i]=vl;

  return;
 }

void MFDenseNVDiff(void *adata,void *bdata, void *cdata, MFErrorHandler e)
 {
  static char RoutineName[]={"MFDenseNVDiff"};
  int i,n;
  struct MFDenseNVectorData *a;
  struct MFDenseNVectorData *b;
  struct MFDenseNVectorData *c;

  a=(struct MFDenseNVectorData*)adata;

#ifdef MFNOCONFIDENCE
  if(a==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for a (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  b=(struct MFDenseNVectorData*)bdata;
  if(b==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for b (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  c=(struct MFDenseNVectorData*)cdata;
  if(c==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for c (argument 3) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(a->nC!=b->nC || a->nC!=c->nC || b->nC!=c->nC)
   {
    sprintf(MFNVectorErrorMsg,"Vectors must all be the same length a=%d, b=%d, c=%d",a->nC,b->nC,c->nC);
    MFSetError(e,4,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  n=a->nC;

  for(i=0;i<n;i++)
   c->data[i]=a->data[i]-b->data[i];

  return;
 }

void MFDenseNVAdd(void *adata,void *bdata,void *cdata, MFErrorHandler e)
 {
  static char RoutineName[]={"MFDenseNVAdd"};
  int i,n;
  struct MFDenseNVectorData *a;
  struct MFDenseNVectorData *b;
  struct MFDenseNVectorData *c;

  a=(struct MFDenseNVectorData*)adata;

#ifdef MFNOCONFIDENCE
  if(a==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for a (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  b=(struct MFDenseNVectorData*)bdata;
  if(b==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for b (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  c=(struct MFDenseNVectorData*)cdata;
  if(c==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for c (argument 3) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(a->nC!=b->nC || a->nC!=c->nC || b->nC!=c->nC)
   {
    sprintf(MFNVectorErrorMsg,"Vectors must all be the same length a=%d, b=%d, c=%d",a->nC,b->nC,c->nC);
    MFSetError(e,4,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  n=a->nC;

  for(i=0;i<n;i++)
   c->data[i]=a->data[i]+b->data[i];

  return;
 }

double *MFNV_CStar(MFNVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"MFNV_CStar"};
  struct MFDenseNVectorData *u;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return NULL;
   }

  if(strcmp(MFNVGetId(this,e),"DENSE"))
   {
    sprintf(MFNVectorErrorMsg,"Trying to get dense arrays from non-dense Vector type \"%s\"",MFNVGetId(this,e));
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  u=(struct MFDenseNVectorData*)MFNVectorGetData(this,e);

  return u->data;
 }

void MFWriteDenseNVectorData(FILE *fid,void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"MFWriteDenseNVectorData"};
  int i;
  struct MFDenseNVectorData *u;

  u=(struct MFDenseNVectorData*)data;

#ifdef MFNOCONFIDENCE
  if(u==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  fprintf(fid,"%d\n",u->nC);
  for(i=0;i<u->nC;i++)
   {
    if(i>0)fprintf(fid," ");
    fprintf(fid,"%lf",u->data[i]);
   }
  fprintf(fid,"\n");

  return;
 }

MFNVector MFReadDenseNVector(FILE *fid, MFErrorHandler e)
 {
  static char RoutineName[]={"MFReadDenseNVector"};
  int n;
  double *vl;
  MFNVector this;
  int i;

  fscanf(fid,"%d\n",&n);

  vl=malloc(n*sizeof(double));

#ifndef MFNOSAFETYNET
  if(n>0&&vl==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Out of memory, trying to allocate %d bytes",n*sizeof(double));
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return;
   }
#endif

  for(i=0;i<n;i++)
   {
    if(i>0)fscanf(fid," ");
    fscanf(fid,"%lf",&(vl[i]));
   }
  fscanf(fid,"\n");

  this=MFCreateNVectorWithData(n,vl,e);

  free(vl);

  return this;
 }

MFNVector MFCloneDenseNVector(void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"MFCloneDenseNVector"};
  struct MFDenseNVectorData *u;

  u=(struct MFDenseNVectorData*)data;

  return MFCreateNVectorWithData(u->nC,u->data,e);
 }

void MFPrintDenseNVector(FILE *fid,void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"MFPrintDenseNVector"};
  int i;
  struct MFDenseNVectorData *u;

  u=(struct MFDenseNVectorData*)data;

#ifdef MFNOCONFIDENCE
  if(fid==NULL)
   {
    sprintf(MFNVectorErrorMsg,"fid (argument 1) is NULL.");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(data==NULL)
   {
    sprintf(MFNVectorErrorMsg,"data (argument 2) is NULL.");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  if(u->nC<1)return;

  if(u->nC<10)
   {
    fprintf(fid,"(%lf",(u->data)[0]);
    for(i=1;i<u->nC;i++)
      fprintf(fid,",%lf",(u->data)[i]);
    fprintf(fid,")");
   }else{
    fprintf(fid,"(%lf",(u->data)[0]);
    for(i=1;i<5;i++)
      fprintf(fid,",%lf",(u->data)[i]);
    fprintf(fid,",...");
    for(i=u->nC-5;i<u->nC;i++)
      fprintf(fid,",%lf",(u->data)[i]);
    fprintf(fid,")");
   }

  return;
 }
