/* 
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *  author: Mike Henderson mhender@watson.ibm.com
 */

static char *id="@(#) $Id: MFChart.c 300 2006-08-31 21:46:08Z mhender $";

#include <MFAtlas.h>
#include <MFChart.h>
#include <MFKVector.h>
#include <MFNVector.h>
#include <MFNKMatrix.h>
#include <MFImplicitMF.h>
#include <MFPrint.h>
#include <MFErrorHandler.h>
#include <stdlib.h>
#include <string.h>

#define MFREALLYTHROWITAWAY

static char MFChartErrorMsg[256]="";

int MFPolytopeTotallyInterior(MFPolytope,MFKVector,double,MFErrorHandler);
int *MFPolytopeVertexIndexSet(MFPolytope,int,MFErrorHandler);
int MFPolytopeVertexNumberOfIndices(MFPolytope,int,MFErrorHandler);

struct MFChartSt {
                  int k;
                  int n;
                  MFImplicitMF M;

                  MFPolytope P;
                  int        allAlphasPositive;
                  int        singular;

                  double     R;
                  double     SuggestR;
                  MFNVector  u;
                  MFNKMatrix Phi;
                  int        posInBList;

                  FILE       *fid;
                  int        paged;
                  int        nearBoundary;
                  int        changed;
                  long       indexInPageFile;

                  int        refNumber;
                  int        nRefs;
                 };

#define DENSE 0
#define AUTO 1
#define LOCA 2

MFChart MFCreateChart(MFImplicitMF M,MFNVector u,MFNKMatrix TS, double R,MFErrorHandler e)
 {
  static char RoutineName[]={"MFCreateChart"};
  MFChart this;

#ifdef MFNOCONFIDENCE
  if(M==NULL)
   {
    sprintf(MFChartErrorMsg,"M (argument 1) is NULL");
    MFSetError(e,8,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(MFIMF_K(M,e)<1)
   {
    sprintf(MFChartErrorMsg,"Manifold has illegal dimension %d. Must be positive",MFIMF_K(M,e));
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(strcmp(MFNVGetId(u,e),"LOCA") && MFIMF_N(M,e)<MFIMF_K(M,e))
   {
    sprintf(MFChartErrorMsg,"Manifold has illegal embedding dimension (%d). Must be greater or equal to it\'s dimension %d.",MFIMF_N(M,e),MFIMF_K(M,e));
    MFSetError(e,12,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(u==NULL)
   {
    sprintf(MFChartErrorMsg,"u (argument 2) is NULL");
    MFSetError(e,8,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this=malloc(sizeof(struct MFChartSt));

#ifndef MFSAFETYNET
  if(this==NULL)
   {
    sprintf(MFChartErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFChartSt));
    MFSetError(e,12,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif

  this->M=M;
  MFRefImplicitMF(M,e);
  this->u=u;
  MFRefNVector(u,e);
  this->n=MFIMF_N(M,e);
  this->k=MFIMF_K(M,e);

  this->Phi=TS;
  MFRefNKMatrix(this->Phi,e);
  this->R=R;
  this->SuggestR=R;
  this->refNumber=0;

  this->P=MFCreateHyperCubeAtOrigin(this->k,1.05*this->R,e);
/*
 *this->P=MFCreateSimplexAtOrigin(this->k,1.05*this->R,e);
 */

#ifdef MFNOCONFIDENCE
  if(TS==NULL)
   {
    sprintf(MFChartErrorMsg,"TangentSpace (argument 3) is NULL ");
    MFSetError(e,12,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return NULL;
   }
#endif

  this->allAlphasPositive=1;
  this->singular=0;
  this->fid=NULL;
  this->paged=0;
  this->nearBoundary=0;
  this->changed=1;
  this->indexInPageFile=-1;
  this->posInBList=-1;
  this->nRefs=1;

  return this;
 }

void MFSubtractHalfSpaceFromChart(MFChart chart,int i,MFKVector n,double o, MFErrorHandler e)
 {
  static char RoutineName[]={"MFSubtractHalfSpaceFromChart"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  MFSubtractHalfSpaceFromPolytope(chart->P,i,n,o,e);
  if(o<0.)chart->allAlphasPositive=0;
  chart->changed=1;
  return;
 }

void MFRefChart(MFChart chart, MFErrorHandler e)
 {
  static char RoutineName[]={"MFRefChart"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  chart->nRefs++;
  return;
 }

void MFFreeChart(MFChart chart, MFErrorHandler e)
 {
  static char RoutineName[]={"MFFreeChart"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  chart->nRefs--;

  if(chart->nRefs>0)return;

  if(chart->M!=NULL){MFFreeImplicitMF(chart->M,e);chart->M=NULL;};
  if(chart->u!=NULL){MFFreeNVector(chart->u,e);chart->u=NULL;};
  if(chart->P!=NULL)
   {
    MFFreePolytope(chart->P,e);
    chart->P=NULL;
   }
  if(chart->Phi!=NULL){MFFreeNKMatrix(chart->Phi,e);chart->Phi=NULL;};
  free(chart);

  return;
 }

MFPolytope MFChartPolytope(MFChart chart, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartPolytope"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  return chart->P;
 }

void MFSetChartPolytope(MFChart chart, MFPolytope P, MFErrorHandler e)
 {
  static char RoutineName[]={"MFSetChartPolytope"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  chart->P=P;

  return;
 }

void MFSetChartRadius(MFChart chart, double R, MFErrorHandler e)
 {
  static char RoutineName[]={"MFSetChartRadius"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  chart->R=R;

  return;
 }

MFNVector MFChartCenter(MFChart chart, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartCenter"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  if(chart->paged)MFChartPageIn(chart,chart->fid,e);

  return chart->u;
 }

MFNKMatrix MFChartTangentSpace(MFChart chart, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartTangentSpace"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  if(chart->paged)MFChartPageIn(chart,chart->fid,e);

  return chart->Phi;
 }

double MFChartRadius(MFChart chart, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartRadius"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  return chart->R;
 }

int MFChartEvaluate(MFChart chart,MFKVector s,MFNVector u, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartEvaluate"};
  MFNVector u0;
  int rc;
  int verbose=0;

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return 0;
   }
#endif

  if(chart->paged)MFChartPageIn(chart,chart->fid,e);

/*u0=MFCloneNVector(u);
#ifdef MFALLOWVERBOSE
  if(verbose)
   {
    fprintf(stderr,"0x%8.8x=u0\n",u0);
    fprintf(stderr,"0x%8.8x=u0(0x%8.8x).data\n",MFNV_CStar(u0,e),u0);
    fflush(stderr);
   }
#endif
  if(chart->paged)MFChartPageIn(chart,chart->fid,e);
  MFChartPointInTangentSpace(chart,s,u0,e);
  rc=MFIMFProject(chart->M,u0,chart->Phi,u,e);
  MFFreeNVector(u0);*/

  rc=MFIMFProjectFromCenter(chart->M,chart->u,chart->Phi,s,u,e);
  return rc;
 }

int MFChartInterior(MFChart chart,MFKVector s, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartInterior"};
  int result;

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  if(MFKVNorm(s,e)>chart->R)
   { 
    return 0;
   }else{
    result=MFPolytopeInterior(chart->P,s,e);
    return result;
   }
 }

void MFChartProjectIntoTangentSpace(MFChart chart,MFNVector u,MFKVector s, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartProjectIntoTangentSpace"};
  MFNVector v;

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(u==NULL)
   {
    sprintf(MFChartErrorMsg,"u (argument 2) is NULL");
    MFSetError(e,8,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(s==NULL)
   {
    sprintf(MFChartErrorMsg,"s (argument 3) is NULL");
    MFSetError(e,8,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(chart->n==0)
   {
    sprintf(MFChartErrorMsg,"Chart has zero dimension");
    MFSetError(e,8,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(chart->paged)MFChartPageIn(chart,chart->fid);

  if(chart->u==NULL)
   {
    sprintf(MFChartErrorMsg,"Chart center is NULL");
    MFSetError(e,8,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  v=MFCloneNVector(u,e);

  MFNSpaceDirection(MFIMFNSpace(chart->M,e),chart->u,u,v,e);
  MFMVMulT(MFIMFNSpace(chart->M,e),chart->Phi,v,s,e);
  MFFreeNVector(v,e);

  return;
 }

void MFChartProjectVectorIntoTangentSpace(MFChart chart,MFNVector u,MFKVector s, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartProjectVectorIntoTangentSpace"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(u==NULL)
   {
    sprintf(MFChartErrorMsg,"u (argument 2) is NULL");
    MFSetError(e,8,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(s==NULL)
   {
    sprintf(MFChartErrorMsg,"s (argument 3) is NULL");
    MFSetError(e,8,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(chart->n==0)
   {
    sprintf(MFChartErrorMsg,"Chart has zero dimension");
    MFSetError(e,8,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(chart->paged)MFChartPageIn(chart,chart->fid);

  if(chart->u==NULL)
   {
    sprintf(MFChartErrorMsg,"Chart center is NULL");
    MFSetError(e,8,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  MFMVMulT(MFIMFNSpace(chart->M,e),chart->Phi,u,s,e);

  return;
 }

void MFChartPointInTangentSpace(MFChart chart,MFKVector s,MFNVector u, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartPointInTangentSpace"};
  MFNVector v;
  int verbose;

#ifdef MFALLOWVERBOSE
  verbose=0;
  if(verbose){printf("%s\n",RoutineName);fflush(stdout);}
#endif

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  if(chart->paged)MFChartPageIn(chart,chart->fid);

  v=MFCloneNVector(u,e);

  MFMVMul(MFIMFNSpace(chart->M,e),chart->Phi,s,v,e);
  MFNSpaceAdd(MFIMFNSpace(chart->M,e),v,chart->u,u,e);
#ifdef MFALLOWVERBOSE
  if(verbose)
   {
    MFPrintNVector(stdout,chart->u,e);
    printf("+");
    MFPrintNKMatrix(stdout,chart->Phi,e);
    MFPrintKVector(stdout,s,e);
    printf("=");
    MFPrintNVector(stdout,u,e);
    printf("\n");
    fflush(stdout);
   }
#endif

  MFFreeNVector(v,e);

  return;
 }

int MFChartHasBoundary(MFChart chart, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartHasBoundary"};
  int vertex,nVertices;

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  nVertices=MFPolytopeNumberOfVertices(chart->P,e);
  for(vertex=0;vertex<nVertices;vertex++)
   {
    if(MFPolytopeRadiusOfVertex(chart->P,vertex,e)>chart->R)
     {
      return 1;
     }
   }
  return 0;
 }

int MFChartK(MFChart chart, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartK"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  return chart->k;
 }

int MFChartN(MFChart chart, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartN"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  return chart->n;
 }

void MFWriteChart(FILE *fid, MFChart chart, MFErrorHandler e)
 {
  static char RoutineName[]={"MFWriteChart"};
  int i;

  if(chart->paged)
   {
/*  MFChartPageIn(chart,chart->fid,e);*/
    fprintf(fid,"%s\n","Chart");
    fprintf(fid,"%d %d %d %d %d %lf %d %d\n",chart->n,chart->k,chart->allAlphasPositive,chart->paged,chart->nearBoundary,chart->R,chart->singular,chart->nRefs,chart->refNumber);
    MFWritePolytope(fid,chart->P,e);
   }else{
    fprintf(fid,"%s\n","Chart",e);
    fprintf(fid,"%d %d %d %d %d %lf %d %d\n",chart->n,chart->k,chart->allAlphasPositive,chart->paged,chart->nearBoundary,chart->R,chart->singular,chart->nRefs,chart->refNumber,e);
    MFWritePolytope(fid,chart->P,e);
    MFWriteNVector(fid,chart->u,e);
    MFWriteNKMatrix(fid,chart->Phi,e);
   }

  return;
 }

MFChart MFReadChart(FILE *fid,MFAtlas A, MFErrorHandler e)
 {
  static char RoutineName[]={"MFReadChart"};
  MFChart chart;
  char tag[100]="";

  fscanf(fid,"%s\n",tag);
#ifdef MFNOCONFIDENCE
  if(strcmp(tag,"Chart"))
   {
    sprintf(MFChartErrorMsg,"Next Object is not a Chart! (%s)\n",RoutineName,tag);
    MFSetError(e,12,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  chart=malloc(sizeof(struct MFChartSt));

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFChartSt));
    MFSetError(e,12,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return NULL;
   }
#endif
  fscanf(fid,"%d %d %d %d %d %lf %d %d\n",&(chart->n),&(chart->k),&(chart->allAlphasPositive),&(chart->paged),&(chart->nearBoundary),&(chart->R),&(chart->singular),&(chart->nRefs),&(chart->refNumber));

  chart->M=MFAtlasMF(A,e);
  chart->P=MFReadPolytope(fid,e);
  if(chart->paged)
   {
    chart->u=NULL;
    chart->Phi=NULL;
   }else{
    chart->u=MFReadNVector(fid,e);
    chart->Phi=MFReadNKMatrix(fid,e);
   }
  chart->posInBList=-1;

  return chart;
 }

int MFChartTotallyInterior(MFChart chart,MFKVector s,double eps, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartTotallyInterior"};
  int result;

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  if(MFKVNorm(s,e)>chart->R)
   { 
    return 0;
   }else{
    result=MFPolytopeTotallyInterior(chart->P,s,eps,e);
    return result;
   }
 }

int MFChartGetPositionInBoundaryList(MFChart this, MFErrorHandler e)
 {
  return this->posInBList;
 }

void MFChartSetPositionInBoundaryList(MFChart this, int pos, MFErrorHandler e)
 {
  this->posInBList=pos;

  return;
 }

MFImplicitMF MFChartGetManifold(MFChart this, MFErrorHandler e)
 {
  return this->M;
 }

int MFChartIsSingular(MFChart this, MFErrorHandler e)
 {
  return this->singular>0;
 }

void MFChartSetSingular(MFChart this, MFErrorHandler e)
 {
  this->singular=1;
  return;
 }

void MFChartSetNonSingular(MFChart this, MFErrorHandler e)
 {
  this->singular=0;
  return;
 }

int MFChartReferenceNumber(MFChart this, MFErrorHandler e)
 {
  return this->refNumber;
 }

void MFChartSetReferenceNumber(MFChart this, int n, MFErrorHandler e)
 {
  this->refNumber=n;
  return;
 }

int MFChartPageOut(MFChart this, FILE *fid, int index, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartPageOut"};
  static int i,j;
  double *c;
  int verbose=0;
  int rc;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFChartErrorMsg,"Chart is NULL");
    MFSetError(e,0,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return 0;
   }
#endif

/* Returns 1 if written, 0 if already paged out */

  if(this->paged)
   {
#ifdef MFALLOWVERBOSE
    if(verbose){printf("Already paged out\n");fflush(stdout);}
#endif
    return 0;
   }

#ifdef MFREALLYPAGE
#ifdef MFALLOWVERBOSE
  if(verbose){fprintf(stderr,"PAGING out chart, fseek to end of file 0x%8.8x\n",fid);fflush(stderr);}
#endif

  rc=fseek(fid,(long)0,SEEK_END);

#ifdef MFALLOWVERBOSE
  if(verbose){fprintf(stderr,"  return code from fseek is %d\n",rc);fprintf(stderr,"ftell= %d\n",ftell(fid));fflush(stderr);}
  if(verbose){fprintf(stderr,"  write to position %d\n",index);fprintf(stderr,"ftell= %d\n",ftell(fid));fflush(stderr);}
#endif

  c=MFNV_CStar(this->u);

#ifdef MFALLOWVERBOSE
  if(verbose){fprintf(stderr,"PAGING   u starts (%lf,%lf,%lf...\n",c[0],c[1],c[2]);fflush(stderr);}
#endif
  this->fid=fid;
  this->paged=1;
  this->indexInPageFile=ftell(fid);
  for(i=0;i<this->n;i++)fprintf(fid,"%lf ",c[i]);
  fprintf(fid,"\n");

  c=MFNKM_CStar(this->Phi,e);  /* BAD What if it's not dense? */

#ifdef MFALLOWVERBOSE
  if(verbose){fprintf(stderr,"PAGING   Phi starts (%lf,%lf,%lf...\n",c[0],c[1],c[2]);fflush(stderr);}
#endif

  for(j=0;j<this->k;j++)
  for(i=0;i<this->n;i++)fprintf(fid,"%lf ",c[i+this->n*j]);
  fprintf(fid,"\n");
#endif

#ifdef MFREALLYTHROWITAWAY
#ifdef MFALLOWVERBOSE
  if(verbose){printf("Discarding contents of chart\n");fflush(stdout);}
#endif

  MFFreeNVector(this->u,e);this->u=NULL;
  MFFreeNKMatrix(this->Phi,e);this->Phi=NULL;
  MFFreePolytope(this->P,e);this->P=NULL;
#endif

  this->paged=1;

  return 1;
 }

int MFChartPageIn(MFChart this, FILE *fid, MFErrorHandler e)
 {
  static int i,j;
  double *c;
  long pos;
  int verbose=0;
  static char RoutineName[]={"MFChartPageIn"};

#ifdef MFALLOWVERBOSE
  if(verbose)
   {
    printf("%s!!!!!\n",RoutineName);
    fflush(stdout);
   }
#endif

  sprintf(MFChartErrorMsg,"Routine %s is broken, FATAL ERROR",RoutineName);
  MFSetError(e,16,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
  return -1;

/* Returns 1 if paged in, 0 if not paged out */

  if(!(this->paged))return 0;

#ifdef MFALLOWVERBOSE
  if(verbose){fprintf(stderr,"PAGING IN CHART! %d in file\n",this->indexInPageFile);fflush(stderr);}
#endif

  if(this->u!=NULL && this->Phi!=NULL)
   {
#ifdef MFALLOWVERBOSE
    if(verbose){fprintf(stderr,"PAGING Already in\n");fflush(stderr);}
#endif
    return 1;
   }

  if(this->u!=NULL)MFFreeNVector(this->u,e);
  this->u=MFCreateNVector(this->n,e);

  if(this->Phi!=NULL)MFFreeNKMatrix(this->Phi,e);
  this->Phi=MFCreateNKMatrixWithData(this->n,this->k,NULL,e);

#ifdef MFALLOWVERBOSE
  if(verbose){fprintf(stderr,"PAGING IN CHART! read from file\n");}
#endif

  fseek(fid,this->indexInPageFile,SEEK_SET);

#ifdef MFALLOWVERBOSE
  if(verbose){fprintf(stderr,"ftell= %d\n",ftell(fid));}
#endif

  c=MFNV_CStar(this->u,e);
  for(i=0;i<this->n;i++)fscanf(fid,"%lf ",&(c[i]));
  fscanf(fid,"\n");

#ifdef MFALLOWVERBOSE
  if(verbose){fprintf(stderr,"PAGING   u starts (%lf,%lf,%lf...\n",c[0],c[1],c[2]);fflush(stderr);}
#endif

  c=MFNKM_CStar(this->Phi,e);
  for(j=0;j<this->k;j++)
  for(i=0;i<this->n;i++)fscanf(fid,"%lf ",&(c[i+this->n*j]));
  fscanf(fid,"\n");

#ifdef MFALLOWVERBOSE
  if(verbose){fprintf(stderr,"PAGING   Phi starts (%lf,%lf,%lf...\n",c[0],c[1],c[2]);fflush(stderr);}
#endif

  fseek(fid,0,SEEK_END);

  return 1;
 }

void MFChartClean(MFChart this, MFErrorHandler e)
 {
  if(!(this->paged))return;

  if(this->u!=NULL){MFFreeNVector(this->u,e);this->u=NULL;}
  if(this->Phi!=NULL){MFFreeNKMatrix(this->Phi,e);this->Phi=NULL;}

  return;
 }

int MFChartPaged(MFChart this, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartPaged"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFChartErrorMsg,"Chart is NULL");
    MFSetError(e,0,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return 0;
   }
#endif

  return this->paged;
 }

int MFChartNearBoundary(MFChart this, MFErrorHandler e)
 {
  return this->nearBoundary;
 }

void MFChartSetNearBoundary(MFChart this,int nearBoundary, MFErrorHandler e)
 {
  this->nearBoundary=nearBoundary;
  return;
 }

int MFChartNumberOfFaces(MFChart this, MFErrorHandler e)
 {
  return MFPolytopeNumberOfFaces(this->P,e);
 }

void MFSetSuggestedChartRadius(MFChart chart, double R, MFErrorHandler e)
 {
  static char RoutineName[]={"MFSetSuggestedChartRadius"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  chart->SuggestR=R;

  return;
 }

double MFChartSuggestedRadius(MFChart chart, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartSuggestedRadius"};

#ifdef MFNOCONFIDENCE
  if(chart==NULL)
   {
    sprintf(MFChartErrorMsg,"Pointer to Chart (argument 1) is NULL");
    MFSetError(e,4,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  return chart->SuggestR;
 }

void MFChartResetChangedFlag(MFChart chart, MFErrorHandler e)
 {
  chart->changed=0;
  return;
 }

int MFChartChangedFlag(MFChart chart, MFErrorHandler e)
 {
  return chart->changed;
 }

void MFChartSetPaged(MFChart this, MFErrorHandler e)
 {
  static char RoutineName[]={"MFChartSetPaged"};

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFChartErrorMsg,"Chart is NULL");
    MFSetError(e,0,RoutineName,MFChartErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  this->paged=1;
  return;
 }
