/*
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *      author: Mike Henderson mhender@watson.ibm.com
 *      date:   February 22, 1999
 */

static char *id="@(#) $Id: MFCSTR.c 257 2006-08-30 20:30:07Z mhender $";

static char MFCSTRMFErrorHandlerMsg[256]="";

extern double MFEpsilon;

#include <MFImplicitMF.h>
#include <MFNKMatrix.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <MFFortran.h>

static void MFFreeCSTRData(void*,MFErrorHandler);
static int MFProjectCSTR(int,int,MFNVector,MFNKMatrix,MFNVector,void*,int*,MFErrorHandler);
static int MFTangentCSTR(int,int,MFNVector,MFNKMatrix,void*,MFErrorHandler);
static int MFTangentCSTRWithGuess(int,int,MFNVector,MFNKMatrix,MFNKMatrix,void*,MFErrorHandler);
static double MFScaleCSTR(int,int,MFNVector,MFNKMatrix,void*,MFErrorHandler);
static void MFWriteCSTRData(FILE*,void*,MFErrorHandler);
static MFImplicitMF MFReadCSTR(FILE*,MFErrorHandler);

static int MFTangentCSTRSingle(int,int,int,double*,double*,double*,double*,void*,MFErrorHandler);
static void MFCurvatureCSTR(int,int,double*,double*,double*,double*,double*,double*,void*,MFErrorHandler);

static double CSTR(double,double,double,double,double,double,double,double,double,double,double,double,double,double,double,MFErrorHandler);
static double TESTdCSTR(double,double,double,double,double,double,double,double,double,double,double,double,double,double,double,
             double,double,double,double,double,double,double,double,double,double,double,double,double,double,double,MFErrorHandler);
static double dCSTR(double,double,double,double,double,double,double,double,double,double,double,double,double,double,double,
             double,double,double,double,double,double,double,double,double,double,double,double,double,double,double,MFErrorHandler);
static double TESTddCSTR(double,double,double,double,double,double,double,double,double,double,double,double,double,double,double,
              double,double,double,double,double,double,double,double,double,double,double,double,double,double,double,
              double,double,double,double,double,double,double,double,double,double,double,double,double,double,double,MFErrorHandler);
static double ddCSTR(double,double,double,double,double,double,double,double,double,double,double,double,double,double,double,
              double,double,double,double,double,double,double,double,double,double,double,double,double,double,double,
              double,double,double,double,double,double,double,double,double,double,double,double,double,double,double,MFErrorHandler);

MFNVector MFNVectorFactory(MFImplicitMF,MFErrorHandler);
MFNKMatrix MFNKMatrixFactory(MFImplicitMF,MFErrorHandler);

struct MFCSTRData
 {
  double q_cr;
  double V_r;
  double qu_o;
  double gama;
  double beta;
  double x10;
  double x20;
  double x30;
  double xi;
  double qu_c;
  double phi;
  double deltaR;
 };

MFImplicitMF MFIMFCreateCSTR(double q_cr,double V_r,double qu_o,double gama,double beta,double x10,double x20,double x30,double xi,double qu_c,double phi,double deltaR, MFErrorHandler e)
 {
  static char RoutineName[]={"MFIMFCreateCSTR"};
  MFImplicitMF cstr;
  int *idata;
  struct MFCSTRData *ddata;
  MFNSpace space;

  cstr=MFIMFCreateBaseClass(3,2,"CSTR",e);

  space=MFCreateNSpace(3,e);
  MFIMFSetSpace(cstr,space,e);
  MFFreeNSpace(space,e);

  ddata=malloc(sizeof(struct MFCSTRData));

#ifndef MFNOSAFETYNET
  if(ddata==NULL)
   {
    sprintf(MFCSTRMFErrorHandlerMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct MFCSTRData));
    MFSetError(e,12,RoutineName,MFCSTRMFErrorHandlerMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    free(cstr);
    return NULL;
   }
#endif

  ddata->q_cr=q_cr;
  ddata->V_r=V_r;
  ddata->qu_o=qu_o;
  ddata->gama=gama;
  ddata->beta=beta;
  ddata->x10=x10;
  ddata->x20=x20;
  ddata->x30=x30;
  ddata->xi=xi;
  ddata->qu_c=qu_c;
  ddata->phi=phi;
  ddata->deltaR=deltaR;

  MFIMFSetData(cstr,(void*)ddata,e);
  MFIMFSetFreeData(cstr,MFFreeCSTRData,e);
  MFIMFSetProject(cstr,MFProjectCSTR,e);
  MFIMFSetTangent(cstr,MFTangentCSTR,e);
  MFIMFSetTangentWithGuess(cstr,MFTangentCSTRWithGuess,e);
  MFIMFSetScale(cstr,MFScaleCSTR,e);
  MFIMFSetWriteData(cstr,MFWriteCSTRData,e);

  MFIMFSetVectorFactory(cstr,MFNVectorFactory,e);
  MFIMFSetMatrixFactory(cstr,MFNKMatrixFactory,e);

  return cstr;
 }

void MFFreeCSTRData(void *d, MFErrorHandler e)
 {
  static char RoutineName[]={"MFFreeCSTRData"};
  free((struct MFCSTRData*)d);
  return;
 }

double MFScaleCSTR(int n,int k,MFNVector vu,MFNKMatrix  mPhi, void *d, MFErrorHandler e)
 {
  static char RoutineName[]={"MFScaleCSTR"};
  static double a00[3],a01[3],a11[3];
  static double A,B,C;
  static double r;
  static double l;
  double *u,*Phi;
  int verbose=0;

  Phi=MFNKM_CStar(mPhi,e);
  u=MFNV_CStar(vu,e);

  if(1)
   {
    MFCurvatureCSTR(n,k,u,Phi,Phi+n,a00,a01,a11,d,e);
    A=sqrt(a00[0]*a00[0]+a00[1]*a00[1]+a00[2]*a00[2]);
    B=sqrt(a01[0]*a01[0]+a01[1]*a01[1]+a01[2]*a01[2]);
    C=sqrt(a11[0]*a11[0]+a11[1]*a11[1]+a11[2]*a11[2]);
    if(A+C>0)
      l=.5*(A+C)+.5*sqrt((A-C)*(A-C)+4*B*B);
     else
      l=-.5*(A+C)+.5*sqrt((A-C)*(A-C)+4*B*B);
    r=sqrt(2*MFEpsilon/l);

#ifdef MFALLOWVERBOSE
    if(verbose)
     {
      printf("%s ev=%lf,%lf, max |ev|=%lf, r=%lf\n",RoutineName,.5*(A+C)+sqrt((A-C)*(A-C)+4*B*B),.5*(A+C)-sqrt((A-C)*(A-C)+4*B*B),l,r);fflush(stdout);
      printf("phi_A=(%lf,%lf,%lf)\n",Phi[0],Phi[1],Phi[2]);fflush(stdout);
      printf("phi_B=(%lf,%lf,%lf)\n",Phi[3],Phi[4],Phi[5]);fflush(stdout);
      printf("phi_A x phi_B=(%lf,%lf,%lf)\n",Phi[2]*Phi[4]-Phi[1]*Phi[5],
                                             Phi[0]*Phi[5]-Phi[2]*Phi[3],
                                             Phi[1]*Phi[3]-Phi[0]*Phi[4]);
      printf("a00=(%lf,%lf,%lf)\n",a00[0],a00[1],a00[2]);fflush(stdout);
      printf("a01=(%lf,%lf,%lf)\n",a01[0],a01[1],a01[2]);fflush(stdout);
      printf("a11=(%lf,%lf,%lf)\n",a11[0],a11[1],a11[2]);fflush(stdout);
      printf("   phiA.phiA=%lf, phiA.phiB=%lf, phiB.phiB=%lf\n",
           Phi[0]*Phi[0]+Phi[1]*Phi[1]+Phi[2]*Phi[2],
           Phi[0]*Phi[3]+Phi[1]*Phi[4]+Phi[2]*Phi[5],
           Phi[3]*Phi[3]+Phi[4]*Phi[4]+Phi[5]*Phi[5]);
      printf("   a00.phiA=%lf, a00.phiB=%lf\n",
           Phi[0]*a00[0]+Phi[1]*a00[1]+Phi[2]*a00[2],
           Phi[3]*a00[0]+Phi[4]*a00[1]+Phi[5]*a00[2]);
      printf("   a01.phiA=%lf, a01.phiB=%lf\n",
           Phi[0]*a01[0]+Phi[1]*a01[1]+Phi[2]*a01[2],
           Phi[3]*a01[0]+Phi[4]*a01[1]+Phi[5]*a01[2]);
      printf("   a11.phiA=%lf, a11.phiB=%lf\n",
           Phi[0]*a11[0]+Phi[1]*a11[1]+Phi[2]*a11[2],
           Phi[3]*a11[0]+Phi[4]*a11[1]+Phi[5]*a11[2]);fflush(stdout);
     }
#endif

    if(r>.1)r=.1;
    if(r<.0001)r=.0001;
   }else
    r=.01;

  return r;
 }

int MFProjectCSTR(int n,int k,MFNVector vu0,MFNKMatrix mPhi,MFNVector vu,void *d,int *index, MFErrorHandler e)
 {
  static char RoutineName[]={"MFProjectCSTR"};
  static int i;
  static double error;
  static double residual;
  static double delta;
  static int itimes;
  static int ierr;
  static double *tangentA;
  static double *tangentB;

  double A[9]={0.,0.,0.,0.,0.,0.,0.,0.,0.};
  double AA[9]={0.,0.,0.,0.,0.,0.,0.,0.,0.};
  double B[3]={0.,0.,0.};
  double BB[3]={0.,0.,0.};
  int    pivots[3]={0,0,0};
  double *u0,*u,*Phi;
  int verbose=0;

  struct MFCSTRData *dd;

#ifdef MFALLOWVERBOSE
  if(verbose){printf("CSTRproject\n");fflush(stdout);}
#endif

  u0=MFNV_CStar(vu0,e);
  Phi=MFNKM_CStar(mPhi,e);
  u=MFNV_CStar(vu,e);

  dd=(struct MFCSTRData*)d;

  tangentA=Phi;
  tangentB=Phi+n;

  for(i=0;i<n;i++)u[i]=u0[i];

  residual=1.;
  itimes=0;
  while(residual>1.e-7)
   {
    A[0]=dCSTR(u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
    A[3]=dCSTR(u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
    A[6]=dCSTR(u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,0.,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);

    A[1]=tangentA[0];
    A[4]=tangentA[1];
    A[7]=tangentA[2];
    A[2]=tangentB[0];
    A[5]=tangentB[1];
    A[8]=tangentB[2];

    B[0]=-CSTR(u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,e);
    B[1]=-tangentA[0]*(u[0]-u0[0])-tangentA[1]*(u[1]-u0[1])-tangentA[2]*(u[2]-u0[2]);
    B[2]=-tangentB[0]*(u[0]-u0[0])-tangentB[1]*(u[1]-u0[1])-tangentB[2]*(u[2]-u0[2]);
    for(i=0;i<9;i++)AA[i]=A[i];
    for(i=0;i<3;i++)BB[i]=B[i];


/* Calculate the size of the right hand side. */

    residual=sqrt(B[0]*B[0]+B[1]*B[1]+B[2]*B[2]);

#ifdef MFALLOWVERBOSE
    if(verbose)
     {
      printf("Correction %d, residual = %lf\n",itimes,residual);
      printf("[ %7.3lf %7.3lf %7.3lf ] [ X[%d] ] = [ %7.3lf ]\n",A[0],A[3],A[6],0,B[0]);
      printf("[ %7.3lf %7.3lf %7.3lf ] [ X[%d] ] = [ %7.3lf ]\n",A[1],A[4],A[7],1,B[1]);
      printf("[ %7.3lf %7.3lf %7.3lf ] [ X[%d] ] = [ %7.3lf ]\n",A[2],A[5],A[8],2,B[2]);
      fflush(stdout);
     }
#endif

    F77_FUNC(dgefa,DGEFA)(A,&n,&n,pivots,&ierr);
    if(ierr!=0)
     {
      printf(" Problem with factor, zero on diagonal %d\n",ierr);
      return 0;
     }
    ierr=0;
    F77_FUNC(dgesl,DGESL)(A,&n,&n,pivots,B,&ierr);

   error=0.;
   error+=fabs(AA[0]*B[0]+AA[3]*B[1]+AA[6]*B[2]-BB[0]);
   error+=fabs(AA[1]*B[0]+AA[4]*B[1]+AA[7]*B[2]-BB[1]);
   error+=fabs(AA[2]*B[0]+AA[5]*B[1]+AA[8]*B[2]-BB[2]);

#ifdef MFALLOWVERBOSE
   if(verbose){printf("  Test solve, %le\n",error);fflush(stdout);}
#endif

/* Calculate the size of the Correction. */


/* Apply the Correction */


    u[0]+=B[0];
    u[1]+=B[1];
    u[2]+=B[2];

    itimes++;
    if(itimes>100)
     {

#ifdef MFALLOWVERBOSE
      if(verbose){printf(" Too many refine iterations %d\n",itimes);fflush(stdout);}
#endif

      return 0;
     }
   }
  if(0&&itimes>5)return 0;

#ifdef MFALLOWVERBOSE
  if(verbose){printf("done %s\n",RoutineName);fflush(stdout);}
#endif

  *index=0;
  return 1;
 }

int MFTangentCSTR(int n,int k,MFNVector vu,MFNKMatrix mPhi,void *d, MFErrorHandler e)
 {
  static char RoutineName[]={"MFTangentCSTR"};
  int i;
  double *Phi0;
  double *u,*Phi;

  Phi=MFNKM_CStar(mPhi,e);
  u=MFNV_CStar(vu,e);

  Phi0=malloc(n*k*sizeof(double));

#ifndef MFNOSAFETYNET
  if(Phi0==NULL)
   {
    sprintf(MFCSTRMFErrorHandlerMsg,"Out of memory, trying to allocate %d bytes",n*k*sizeof(double));
    MFSetError(e,12,RoutineName,MFCSTRMFErrorHandlerMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return;
   }
#endif

  for(i=0;i<n*k;i++)Phi0[i]=0.;
  Phi0[1]=1.;
  Phi0[5]=1.;
   
  MFTangentCSTRSingle(n,k,0,u,Phi0,Phi0+n,Phi,d,e);
  MFTangentCSTRSingle(n,k,1,u,Phi ,Phi0+n,Phi,d,e);
  free(Phi0);
  return 1;
 }

int MFTangentCSTRWithGuess(int n,int k,MFNVector vu,MFNKMatrix mPhi0,MFNKMatrix mPhi,void *d, MFErrorHandler e)
 {
  static char RoutineName[]={"MFTangentCSTRWithGuess"};
  double *u,*Phi,*Phi0;

  Phi0=MFNKM_CStar(mPhi0,e);
  Phi=MFNKM_CStar(mPhi,e);
  u=MFNV_CStar(vu,e);
  MFTangentCSTRSingle(n,k,0,u,Phi0,Phi0+n,Phi,d,e);
  MFTangentCSTRSingle(n,k,1,u,Phi ,Phi0+n,Phi,d,e);
  return 1;
 }

int MFTangentCSTRSingle(int n,int k,int t, double *u,double *tangentA,double *tangentB,double *Phi, void *d, MFErrorHandler e)
{
  static int ierr;
  static double s;

  double A[9]={0.,0.,0.,0.,0.,0.,0.,0.,0.};
  double B[3]={0.,0.,0.};
  int    pivots[3]={0,0,0};
  struct MFCSTRData *dd;

  dd=(struct MFCSTRData*)d;

  A[0]=dCSTR(u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  A[3]=dCSTR(u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  A[6]=dCSTR(u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,0.,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);

  A[1]=tangentA[0];
  A[4]=tangentA[1];
  A[7]=tangentA[2];
  A[2]=tangentB[0];
  A[5]=tangentB[1];
  A[8]=tangentB[2];

  B[0]=0.;
  B[1]=0.;
  B[2]=0.;
  B[1+t]=1.;

  F77_FUNC(dgefa,DGEFA)(A,&n,&n,pivots,&ierr);
  if(ierr!=0)
   {
    printf(" Problem with factor, zero on diagonal %d\n",ierr);
    exit(8);
   }
  ierr=0;
  F77_FUNC(dgesl,DGESL)(A,&n,&n,pivots,B,&ierr);


/* Normalize */

  s=1./sqrt(B[0]*B[0]+B[1]*B[1]+B[2]*B[2]);
  Phi[0+n*t]=s*B[0];
  Phi[1+n*t]=s*B[1];
  Phi[2+n*t]=s*B[2];

  return 1;
 }

double CSTR(double u1,double u2,double x2,double q_cr,double V_r,double qu_o,double gama,double beta,double x10,double x20,double x30,double xi,double qu_c,double phi,double deltaR, MFErrorHandler e)
 {
  static char RoutineName[]={"CSTR"};
  double CSTR;

  CSTR = qu_o*(x20-x2) + (beta*phi*exp(gama*x2/(1+x2))*u2*qu_o*x10)/(qu_o+phi*exp(gama*x2/(1+x2))*u2)-(deltaR*u2*qu_c*u1)*(x2-x30)/(qu_c*u1+xi*deltaR*u2);
  return CSTR;
 }

double ddCSTR(double  u1,double  u2,double  x2,double  q_cr,double  V_r,double  qu_o,double  gama,double  beta,double  x10,double  x20,double  x30,double  xi,double  qu_c,double  phi,double  deltaR,
             double dAu1,double dAu2,double dAx2,double dAq_cr,double dAV_r,double dAqu_o,double dAgama,double dAbeta,double dAx10,double dAx20,double dAx30,double dAxi,double dAqu_c,double dAphi,double dAdeltaR,
             double dBu1,double dBu2,double dBx2,double dBq_cr,double dBV_r,double dBqu_o,double dBgama,double dBbeta,double dBx10,double dBx20,double dBx30,double dBxi,double dBqu_c,double dBphi,double dBdeltaR, MFErrorHandler e)
 {
  static char RoutineName[]={"ddCSTR"};
  double ddCSTR;
  double approx;
  double   t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
  double  dAt1,dAt2,dAt3,dAt4,dAt5,dAt6,dAt7,dAt8,dAt9,dAt10;
  double  dBt1,dBt2,dBt3,dBt4,dBt5,dBt6,dBt7,dBt8,dBt9,dBt10;
  double dAdBt1,dAdBt2,dAdBt3,dAdBt4,dAdBt5,dAdBt6,dAdBt7,dAdBt8,dAdBt9,dAdBt10;

      t1= gama* x2/(1+x2);
    dAt1=dAgama*  x2/(1+x2)
          +gama*dAx2/(1+x2)
          -gama*  x2*dAx2/(1+x2)/(1+x2);
    dBt1=dBgama*  x2/(1+x2)
          +gama*dBx2/(1+x2)
          -gama*  x2*dBx2/(1+x2)/(1+x2);
  dAdBt1=dAgama*dBx2/(1+x2)
        -dAgama*  x2*dBx2/(1+x2)/(1+x2)
        +dBgama*dAx2/(1+x2)
        -dBgama*dAx2*dBx2/(1+x2)/(1+x2)
          -gama*dBx2*dAx2/(1+x2)/(1+x2)
        +2*gama*  x2*dAx2*dBx2/(1+x2)/(1+x2)/(1+x2);

      t2=exp(t1);
    dAt2=t2*dAt1;
    dBt2=t2*dBt1;
  dAdBt2=dBt2*dAt1;

      t3=qu_o*(x20-x2);
    dAt3=dAqu_o*(x20-x2)+qu_o*(dAx20-dAx2);
    dBt3=dBqu_o*(x20-x2)+qu_o*(dBx20-dBx2);
  dAdBt3=dAqu_o*(dBx20-dBx2)+dBqu_o*(dAx20-dAx2);

        t4=beta*phi*t2*u2*qu_o*x10;
      dAt4=dAbeta*phi*t2*u2*qu_o*x10
          +beta*dAphi*t2*u2*qu_o*x10
          +beta*phi*dAt2*u2*qu_o*x10
          +beta*phi*t2*dAu2*qu_o*x10
          +beta*phi*t2*u2*dAqu_o*x10
          +beta*phi*t2*u2*qu_o*dAx10;
      dBt4=dBbeta*phi*t2*u2*qu_o*x10
          +beta*dBphi*t2*u2*qu_o*x10
          +beta*phi*dBt2*u2*qu_o*x10
          +beta*phi*t2*dBu2*qu_o*x10
          +beta*phi*t2*u2*dBqu_o*x10
          +beta*phi*t2*u2*qu_o*dBx10;
    dAdBt4=dAbeta*dBphi*t2*u2*qu_o*x10
          +dAbeta*phi*dBt2*u2*qu_o*x10
          +dAbeta*phi*t2*dBu2*qu_o*x10
          +dAbeta*phi*t2*u2*dBqu_o*x10
          +dAbeta*phi*t2*u2*qu_o*dBx10
          +dBbeta*dAphi*t2*u2*qu_o*x10
          +beta*dAphi*dBt2*u2*qu_o*x10
          +beta*dAphi*t2*dBu2*qu_o*x10
          +beta*dAphi*t2*u2*dBqu_o*x10
          +beta*dAphi*t2*u2*qu_o*dBx10
          +dBbeta*phi*dAt2*u2*qu_o*x10
          +beta*dBphi*dAt2*u2*qu_o*x10
          +beta*phi*dAt2*dBu2*qu_o*x10
          +beta*phi*dAt2*u2*dBqu_o*x10
          +beta*phi*dAt2*u2*qu_o*dBx10
          +dBbeta*phi*t2*dAu2*qu_o*x10
          +beta*dBphi*t2*dAu2*qu_o*x10
          +beta*phi*dBt2*dAu2*qu_o*x10
          +beta*phi*t2*dAu2*dBqu_o*x10
          +beta*phi*t2*dAu2*qu_o*dBx10
          +dBbeta*phi*t2*u2*dAqu_o*x10
          +beta*dBphi*t2*u2*dAqu_o*x10
          +beta*phi*dBt2*u2*dAqu_o*x10
          +beta*phi*t2*dBu2*dAqu_o*x10
          +beta*phi*t2*u2*dAqu_o*dBx10
          +dBbeta*phi*t2*u2*qu_o*dAx10
          +beta*dBphi*t2*u2*qu_o*dAx10
          +beta*phi*dBt2*u2*qu_o*dAx10
          +beta*phi*t2*dBu2*qu_o*dAx10
          +beta*phi*t2*u2*dBqu_o*dAx10;

      t5=qu_o
        +phi*t2*u2;
    dAt5=dAqu_o
        +dAphi*t2*u2
        +phi*dAt2*u2
        +phi*t2*dAu2;
    dBt5=dBqu_o
        +dBphi*t2*u2
        +phi*dBt2*u2
        +phi*t2*dBu2;
   dAdBt5=dAphi*dBt2*u2
         +dAphi*t2*dBu2
         +dBphi*dAt2*u2
         +phi*dAt2*dBu2
         +dBphi*t2*dAu2;
         +phi*dBt2*dAu2;

      t6=deltaR*u2*qu_c*u1;
    dAt6=dAdeltaR*u2*qu_c*u1
        +deltaR*dAu2*qu_c*u1
        +deltaR*u2*dAqu_c*u1
        +deltaR*u2*qu_c*dAu1;
    dBt6=dBdeltaR*u2*qu_c*u1
        +deltaR*dBu2*qu_c*u1
        +deltaR*u2*dBqu_c*u1
        +deltaR*u2*qu_c*dBu1;
  dAdBt6=dAdeltaR*dBu2*qu_c*u1
        +dAdeltaR*u2*dBqu_c*u1
        +dAdeltaR*u2*qu_c*dBu1
        +dBdeltaR*dAu2*qu_c*u1
        +deltaR*dAu2*dBqu_c*u1
        +deltaR*dAu2*qu_c*dBu1
        +dBdeltaR*u2*dAqu_c*u1
        +deltaR*dBu2*dAqu_c*u1
        +deltaR*u2*dAqu_c*dBu1
        +dBdeltaR*u2*qu_c*dAu1
        +deltaR*dBu2*qu_c*dAu1
        +deltaR*u2*dBqu_c*dAu1;

      t7=x2-x30;
    dAt7=dAx2-dAx30;
    dBt7=dBx2-dBx30;
  dAdBt7=0.;

     t8=qu_c*u1
       +xi*deltaR*u2;
   dAt8=dAqu_c*u1
       +qu_c*dAu1
       +dAxi*deltaR*u2
       +xi*dAdeltaR*u2
       +xi*deltaR*dAu2;
   dBt8=dBqu_c*u1
       +qu_c*dBu1
       +dBxi*deltaR*u2
       +xi*dBdeltaR*u2
       +xi*deltaR*dBu2;
  dAdBt8=dAqu_c*dBu1
        +dBqu_c*dAu1
        +dAxi*dBdeltaR*u2
        +dAxi*deltaR*dBu2
        +dBxi*dAdeltaR*u2
        +xi*dAdeltaR*dBu2
        +dBxi*deltaR*dAu2
        +xi*dBdeltaR*dAu2;

      t9=t4/t5;
    dAt9=dAt4/t5
        -t4*dAt5/t5/t5;
    dBt9=dBt4/t5
        -t4*dBt5/t5/t5;
  dAdBt9=-dAt4*dBt5/t5/t5
         -dBt4*dAt5/t5/t5;
         +2*t4*dAt5*dBt5/t5/t5/t5;

      t10=-t6*t7/t8;
    dAt10=-dAt6*t7/t8
          -t6*dAt7/t8
          +t6*t7*dAt8/t8/t8;
    dBt10=-dBt6*t7/t8
          -t6*dBt7/t8
          +t6*t7*dBt8/t8/t8;
  dAdBt10=-dAt6*dBt7/t8
          +dAt6*t7*dBt8/t8/t8
          -dBt6*dAt7/t8
          +t6*dAt7*dBt8/t8/t8
          +dBt6*t7*dAt8/t8/t8
          +t6*dBt7*dAt8/t8/t8
          -2*t6*t7*dAt8*dBt8/t8/t8/t8;

  ddCSTR = dAdBt3 + dAdBt9 + dAdBt10;

/*
  printf("ddCSTR=%21.14le+%21.14le+%21.14le=%21.14le\n",dAdBt3,dAdBt9,dAdBt10,ddCSTR);fflush(stdout);

  printf("ddCSTR: %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf\n",
     u1,u2,x2,q_cr,V_r,qu_o,gama,beta,x10,x20,x30,xi,qu_c,phi,deltaR);
  printf("       %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf\n",
     dAu1,dAu2,dAx2,dAq_cr,dAV_r,dAqu_o,dAgama,dAbeta,dAx10,dAx20,dAx30,dAxi,dAqu_c,dAphi,dAdeltaR);
  printf("       %21.14lf %21.14lf %21.14lf %21.14lf %21.14lf %21.14lf %21.14lf %21.14lf %21.14lf %21.14lf %21.14lf %21.14lf %21.14lf %21.14lf %21.14lf\n",
     dBu1,dBu2,dBx2,dBq_cr,dBV_r,dBqu_o,dBgama,dBbeta,dBx10,dBx20,dBx30,dBxi,dBqu_c,dBphi,dBdeltaR);
  printf("       t1 =%21.14lf dAt1 =%21.14lf dBt1 =%21.14lf dAdBt1=%21.14lf\n",t1,dAt1,dBt1,dAdBt1);
  printf("       t2 =%21.14lf dAt2 =%21.14lf dBt2 =%21.14lf dAdBt2=%21.14lf\n",t2,dAt2,dBt2,dAdBt2);
  printf("       t3 =%21.14lf dAt3 =%21.14lf dBt3 =%21.14lf dAdBt3=%21.14lf\n",t3,dAt3,dBt3,dAdBt3);
  printf("       t4 =%21.14lf dAt4 =%21.14lf dBt4 =%21.14lf dAdBt4=%21.14lf\n",t4,dAt4,dBt4,dAdBt4);
  printf("       t5 =%21.14lf dAt5 =%21.14lf dBt5 =%21.14lf dAdBt5=%21.14lf\n",t5,dAt5,dBt5,dAdBt5);
  printf("       t6 =%21.14lf dAt6 =%21.14lf dBt6 =%21.14lf dAdBt6=%21.14lf\n",t6,dAt6,dBt6,dAdBt6);
  printf("       t7 =%21.14lf dAt7 =%21.14lf dBt7 =%21.14lf dAdBt7=%21.14lf\n",t7,dAt7,dBt7,dAdBt7);
  printf("       t8 =%21.14lf dAt8 =%21.14lf dBt8 =%21.14lf dAdBt8=%21.14lf\n",t8,dAt8,dBt8,dAdBt8);
  printf("       t9 =%21.14lf dAt9 =%21.14lf dBt9 =%21.14lf dAdBt9=%21.14lf\n",t9,dAt9,dBt9,dAdBt9);
  printf("       t10=%21.14lf dAt10=%21.14lf dBt10=%21.14lf dAdBt10=%21.14lf\n",t10,dAt10,dBt10,dAdBt10);
  printf("    ddCSTR=%21.14lf\n",ddCSTR);

  approx=TESTddCSTR(u1,u2,x2,q_cr,V_r,qu_o,gama,beta,x10,x20,x30,xi,qu_c,phi,deltaR,
             dAu1,dAu2,dAx2,dAq_cr,dAV_r,dAqu_o,dAgama,dAbeta,dAx10,dAx20,dAx30,dAxi,dAqu_c,dAphi,dAdeltaR,
             dBu1,dBu2,dBx2,dBq_cr,dBV_r,dBqu_o,dBgama,dBbeta,dBx10,dBx20,dBx30,dBxi,dBqu_c,dBphi,dBdeltaR);
  printf("ddCSTR: code %le, diff %le, Error %le\n",
         ddCSTR,approx,fabs(ddCSTR-approx));fflush(stdout);
*/

  return ddCSTR;
 }

void MFCurvatureCSTR(int n,int k, double *u,double *tangentA,double *tangentB,double *a00, double *a01, double *a11, void *d, MFErrorHandler e)
{
  static char RoutineName[]={"MFCurvatureCSTR"};
  static int ierr;

  static double A[9],B[3];
  static double Guu[9];
  static int    pivots[3];
  struct MFCSTRData *dd;
  int verbose=0;

  dd=(struct MFCSTRData*)d;

  A[0]=dCSTR(u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  A[3]=dCSTR(u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  A[6]=dCSTR(u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,0.,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);

  A[1]=tangentA[0];
  A[4]=tangentA[1];
  A[7]=tangentA[2];
  A[2]=tangentB[0];
  A[5]=tangentB[1];
  A[8]=tangentB[2];

#ifdef MFALLOWVERBOSE
  if(verbose)
   {
    printf("Curvature\n");
    printf("[ %7.3lf %7.3lf %7.3lf ]\n",A[0],A[3],A[6]);
    printf("[ %7.3lf %7.3lf %7.3lf ]\n",A[1],A[4],A[7]);
    printf("[ %7.3lf %7.3lf %7.3lf ]\n",A[2],A[5],A[8]);
    fflush(stdout);
   }
#endif

  F77_FUNC(dgefa,DGEFA)(A,&n,&n,pivots,&ierr);
  if(ierr!=0)
   {
    printf(" Problem with factor, zero on diagonal %d\n",ierr);
    exit(8);
   }

  Guu[0]=ddCSTR( u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,
               1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
               1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  Guu[1]=ddCSTR( u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,
               1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
               0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  Guu[2]=ddCSTR( u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,
               1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
               0.,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  Guu[3]=ddCSTR( u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,
               0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
               1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  Guu[4]=ddCSTR( u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,
               0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
               0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  Guu[5]=ddCSTR( u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,
               0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
               0.,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  Guu[6]=ddCSTR( u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,
               0.,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
               1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  Guu[7]=ddCSTR( u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,
               0.,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
               0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);
  Guu[8]=ddCSTR( u[0],u[1],u[2],dd->q_cr,dd->V_r,dd->qu_o,dd->gama,dd->beta,dd->x10,dd->x20,dd->x30,dd->xi,dd->qu_c,dd->phi,dd->deltaR,
               0.,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
               0.,0.,1.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,e);

#ifdef MFALLOWVERBOSE
  if(verbose)
   {
    printf("Guu\n");
    printf("[ %7.3lf %7.3lf %7.3lf ]\n",Guu[0],Guu[3],Guu[6]);
    printf("[ %7.3lf %7.3lf %7.3lf ]\n",Guu[1],Guu[4],Guu[7]);
    printf("[ %7.3lf %7.3lf %7.3lf ]\n",Guu[2],Guu[5],Guu[8]);
    fflush(stdout);
   }
#endif

  a01[0]=-Guu[0]*tangentA[0]*tangentB[0]
         -Guu[1]*tangentA[0]*tangentB[1]
         -Guu[2]*tangentA[0]*tangentB[2]
         -Guu[3]*tangentA[1]*tangentB[0]
         -Guu[4]*tangentA[1]*tangentB[1]
         -Guu[5]*tangentA[1]*tangentB[2]
         -Guu[6]*tangentA[2]*tangentB[0]
         -Guu[7]*tangentA[2]*tangentB[1]
         -Guu[8]*tangentA[2]*tangentB[2];

#ifdef MFALLOWVERBOSE
  if(verbose)printf(" Guu PhiAPhiB %7.3lf\n",a01[0]);
#endif

  a01[1]=0.;
  a01[2]=0.;
  ierr=0;
  F77_FUNC(dgesl,DGESL)(A,&n,&n,pivots,a01,&ierr);

  a11[0]=-Guu[0]*tangentB[0]*tangentB[0]
         -Guu[1]*tangentB[0]*tangentB[1]
         -Guu[2]*tangentB[0]*tangentB[2]
         -Guu[3]*tangentB[1]*tangentB[0]
         -Guu[4]*tangentB[1]*tangentB[1]
         -Guu[5]*tangentB[1]*tangentB[2]
         -Guu[6]*tangentB[2]*tangentB[0]
         -Guu[7]*tangentB[2]*tangentB[1]
         -Guu[8]*tangentB[2]*tangentB[2];

#ifdef MFALLOWVERBOSE
  if(verbose)printf(" Guu PhiBPhiB %7.3lf\n",a11[0]);
#endif

  a11[1]=0.;
  a11[2]=0.;
  ierr=0;
  F77_FUNC(dgesl,DGESL)(A,&n,&n,pivots,a11,&ierr);

  return;
 }

double TESTdCSTR(double  u1,double  u2,double  x2,double  q_cr,double  V_r,double  qu_o,double  gama,double  beta,double  x10,double  x20,double  x30,double  xi,double  qu_c,double  phi,double  deltaR,
             double du1,double du2,double dx2,double dq_cr,double dV_r,double dqu_o,double dgama,double dbeta,double dx10,double dx20,double dx30,double dxi,double dqu_c,double dphi,double ddeltaR, MFErrorHandler err)
 {
  static char RoutineName[]={"TESTdCSTR"};
  double e=1.e-10;
  double CSTRP;
  double CSTRM;
  double approx;

  CSTRP=CSTR(u1    +e*du1,
             u2    +e*du2,
             x2    +e*dx2,
             q_cr  +e*dq_cr,
             V_r   +e*dV_r,
             qu_o  +e*dqu_o,
             gama  +e*dgama,
             beta  +e*dbeta,
             x10   +e*dx10,
             x20   +e*dx20,
             x30   +e*dx30,
             xi    +e*dxi,
             qu_c  +e*dqu_c,
             phi   +e*dphi,
             deltaR+e*ddeltaR,err);
  CSTRM=CSTR(u1    -e*du1,
             u2    -e*du2,
             x2    -e*dx2,
             q_cr  -e*dq_cr,
             V_r   -e*dV_r,
             qu_o  -e*dqu_o,
             gama  -e*dgama,
             beta  -e*dbeta,
             x10   -e*dx10,
             x20   -e*dx20,
             x30   -e*dx30,
             xi    -e*dxi,
             qu_c  -e*dqu_c,
             phi   -e*dphi,
             deltaR-e*ddeltaR,err);

  approx=.5*(CSTRP-CSTRM)/e;

  return approx;
 }

void MFWriteCSTRData(FILE *fid,void *d, MFErrorHandler e)
 {
  static char RoutineName[]={"MFWriteCSTRData"};
  struct MFCSTRData *data;

  data=(struct MFCSTRData*)d;

  fprintf(fid,"%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf\n",
  data->q_cr,data->V_r,data->qu_o,data->gama,data->beta,data->x10,data->x20,data->x30,data->xi,data->qu_c,data->phi,data->deltaR);

  fflush(fid);
  return;
 }

MFImplicitMF MFReadCSTR(FILE *fid, MFErrorHandler e)
 {
  static char RoutineName[]={"MFIMFReadCSTR"};
  double q_cr=0.;
  double V_r=0.;
  double qu_o=0.;
  double gama=0.;
  double beta=0.;
  double x10=0.;
  double x20=0.;
  double x30=0.;
  double xi=0.;
  double qu_c=0.;
  double phi=0.;
  double deltaR=0.;
  MFImplicitMF cstr;
  int *idata;
  struct MFCSTRData *ddata;

  fscanf(fid,"%lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf %lf\n",&q_cr,&V_r,&qu_o,&gama,&beta,&x10,&x20,&x30,&xi,&qu_c,&phi,&deltaR);
  cstr=MFIMFCreateCSTR(q_cr,V_r,qu_o,gama,beta,x10,x20,x30,xi,qu_c,phi,deltaR,e);

  return cstr;
 }

double TESTddCSTR(double  u1,double  u2,double  x2,double  q_cr,double  V_r,double  qu_o,double  gama,double  beta,double  x10,double  x20,double  x30,double  xi,double  qu_c,double  phi,double  deltaR,
             double dAu1,double dAu2,double dAx2,double dAq_cr,double dAV_r,double dAqu_o,double dAgama,double dAbeta,double dAx10,double dAx20,double dAx30,double dAxi,double dAqu_c,double dAphi,double dAdeltaR,
             double dBu1,double dBu2,double dBx2,double dBq_cr,double dBV_r,double dBqu_o,double dBgama,double dBbeta,double dBx10,double dBx20,double dBx30,double dBxi,double dBqu_c,double dBphi,double dBdeltaR, MFErrorHandler err)
 {
  static char RoutineName[]={"TESTddCSTR"};
  double e=1.e-10;
  double dCSTRP;
  double dCSTRM;
  double approx;

  dCSTRP=dCSTR(u1    +e*dBu1,
               u2    +e*dBu2,
               x2    +e*dBx2,
               q_cr  +e*dBq_cr,
               V_r   +e*dBV_r,
               qu_o  +e*dBqu_o,
               gama  +e*dBgama,
               beta  +e*dBbeta,
               x10   +e*dBx10,
               x20   +e*dBx20,
               x30   +e*dBx30,
               xi    +e*dBxi,
               qu_c  +e*dBqu_c,
               phi   +e*dBphi,
               deltaR+e*dBdeltaR,
               dAu1,
               dAu2,
               dAx2,
               dAq_cr,
               dAV_r,
               dAqu_o,
               dAgama,
               dAbeta,
               dAx10,
               dAx20,
               dAx30,
               dAxi,
               dAqu_c,
               dAphi,
               dAdeltaR,err);

  dCSTRM=dCSTR(u1    -e*dBu1,
               u2    -e*dBu2,
               x2    -e*dBx2,
               q_cr  -e*dBq_cr,
               V_r   -e*dBV_r,
               qu_o  -e*dBqu_o,
               gama  -e*dBgama,
               beta  -e*dBbeta,
               x10   -e*dBx10,
               x20   -e*dBx20,
               x30   -e*dBx30,
               xi    -e*dBxi,
               qu_c  -e*dBqu_c,
               phi   -e*dBphi,
               deltaR-e*dBdeltaR,
               dAu1,
               dAu2,
               dAx2,
               dAq_cr,
               dAV_r,
               dAqu_o,
               dAgama,
               dAbeta,
               dAx10,
               dAx20,
               dAx30,
               dAxi,
               dAqu_c,
               dAphi,
               dAdeltaR,err);

  approx=.5*(dCSTRP-dCSTRM)/e;

  return approx;
 }

double dCSTR(double  u1,double  u2,double  x2,double  q_cr,double  V_r,double  qu_o,double  gama,double  beta,double  x10,double  x20,double  x30,double  xi,double  qu_c,double  phi,double  deltaR,
             double du1,double du2,double dx2,double dq_cr,double dV_r,double dqu_o,double dgama,double dbeta,double dx10,double dx20,double dx30,double dxi,double dqu_c,double dphi,double ddeltaR, MFErrorHandler e)
 {
  static char RoutineName[]={"dCSTR"};
  double dCSTR;
  double approx;
  double  t1, t2, t3, t4, t5, t6, t7, t8, t9, t10;
  double dt1,dt2,dt3,dt4,dt5,dt6,dt7,dt8,dt9,dt10;

   t1=gama*x2/(1+x2);
  dt1=dgama*x2/(1+x2)+gama*dx2/(1+x2)-gama*x2*dx2/(1+x2)/(1+x2);

   t2=exp(t1);
  dt2=t2*dt1;

   t3=qu_o*(x20-x2);
  dt3=dqu_o*(x20-x2)+qu_o*(dx20-dx2);

   t4=beta*phi*t2*u2*qu_o*x10;
  dt4=dbeta*phi*t2*u2*qu_o*x10+beta*dphi*t2*u2*qu_o*x10+beta*phi*dt2*u2*qu_o*x10+beta*phi*t2*du2*qu_o*x10+beta*phi*t2*u2*dqu_o*x10+beta*phi*t2*u2*qu_o*dx10;

   t5=qu_o+phi*t2*u2;
  dt5=dqu_o+dphi*t2*u2+phi*dt2*u2+phi*t2*du2;

   t6=deltaR*u2*qu_c*u1;
  dt6=ddeltaR*u2*qu_c*u1+deltaR*du2*qu_c*u1+deltaR*u2*dqu_c*u1+deltaR*u2*qu_c*du1;

   t7=x2-x30;
  dt7=dx2-dx30;

   t8=qu_c*u1+xi*deltaR*u2;
  dt8=dqu_c*u1+qu_c*du1+dxi*deltaR*u2+xi*ddeltaR*u2+xi*deltaR*du2;

   t9=t4/t5;
  dt9=dt4/t5-t4*dt5/t5/t5;

   t10=-t6*t7/t8;
  dt10=-dt6*t7/t8-t6*dt7/t8+t6*t7*dt8/t8/t8;

  dCSTR = dt3 + dt9 + dt10;

/*
  printf("dCSTR=%21.14le+%21.14le+%21.14le=%21.14le\n",dt3,dt9,dt10,dCSTR);fflush(stdout);

  printf("dCSTR: %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf\n",
     u1,u2,x2,q_cr,V_r,qu_o,gama,beta,x10,x20,x30,xi,qu_c,phi,deltaR);
  printf("       %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf %7.3lf\n",
     du1,du2,dx2,dq_cr,dV_r,dqu_o,dgama,dbeta,dx10,dx20,dx30,dxi,dqu_c,dphi,ddeltaR);
  printf("       t1 =%21.14lf dt1 =%21.14lf\n",t1,dt1);
  printf("       t2 =%21.14lf dt2 =%21.14lf\n",t2,dt2);
  printf("       t3 =%21.14lf dt3 =%21.14lf\n",t3,dt3);
  printf("       t4 =%21.14lf dt4 =%21.14lf\n",t4,dt4);
  printf("       t5 =%21.14lf dt5 =%21.14lf\n",t5,dt5);
  printf("       t6 =%21.14lf dt6 =%21.14lf\n",t6,dt6);
  printf("       t7 =%21.14lf dt7 =%21.14lf\n",t7,dt7);
  printf("       t8 =%21.14lf dt8 =%21.14lf\n",t8,dt8);
  printf("       t9 =%21.14lf dt9 =%21.14lf\n",t9,dt9);
  printf("       t10=%21.14lf dt10=%21.14lf\n",t10,dt10);
  printf("     dCSTR=%21.14lf\n",dCSTR);

  approx=TESTdCSTR(u1,u2,x2,q_cr,V_r,qu_o,gama,beta,x10,x20,x30,xi,qu_c,phi,deltaR,
             du1,du2,dx2,dq_cr,dV_r,dqu_o,dgama,dbeta,dx10,dx20,dx30,dxi,dqu_c,dphi,ddeltaR,e);
  printf("TESTdCSTR: code %le, diff %le, Error %le\n",
         dCSTR,approx,fabs(dCSTR-approx));fflush(stdout);
*/

  return dCSTR;
 }
