/*
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *      author: Mike Henderson mhender@watson.ibm.com
 *      date:   March 1, 1999
 */

static char *id="@(#) $Id: IMFExpansionSpace.c 242 2006-08-15 21:16:26Z mhender $";

static char IMFExpansionSpaceErrorMsg[256]="";

#include <MFNSpace.h>
#include <MFNVector.h>
#include <MFErrorHandler.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

struct IMFExpansionSpaceData
 {
  int n;
 };

static void MFFreeExpansionSpaceData(void*,MFErrorHandler);
static double IMFExpansionSpaceDistance(MFNSpace,MFNVector,MFNVector,void*,MFErrorHandler);
static void IMFExpansionSpaceDirection(MFNSpace,MFNVector,MFNVector,MFNVector,void*,MFErrorHandler);
static void IMFExpansionSpaceAdd(MFNSpace,MFNVector,MFNVector,MFNVector,void*,MFErrorHandler);
static void IMFExpansionSpaceScale(MFNSpace,double,MFNVector,MFNVector,void*,MFErrorHandler);
static double IMFExpansionSpaceInner(MFNSpace,MFNVector,MFNVector,void*,MFErrorHandler);

/*! \fn MFNSpace IMFCreateExpansionSpace(int n);
 *  \brief An MFNSpace defining operations on a point on a fat trajectory.
 *
 *  \param n The dimension of the embedding space of the expansion.
 *  \returns A new IMFExpansionNSpace.
 */
MFNSpace IMFCreateExpansionSpace(int n, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFCreateExpansionSpace"};
  MFNSpace this;
  struct IMFExpansionSpaceData *data;

  this=MFCreateNSpaceBaseClass("ExpansionSpace",e);
  MFNSpaceSetDistance(this,IMFExpansionSpaceDistance,e);
  MFNSpaceSetInnerProduct(this,IMFExpansionSpaceInner,e);
  MFNSpaceSetDirection(this,IMFExpansionSpaceDirection,e);
  MFNSpaceSetAdd(this,IMFExpansionSpaceAdd,e);
  MFNSpaceSetScale(this,IMFExpansionSpaceScale,e);
  MFNSpaceSetFreeData(this,MFFreeExpansionSpaceData,e);
  data=malloc(sizeof(struct IMFExpansionSpaceData));

#ifndef MFNOSAFETYNET
  if(data==NULL)
   {
    sprintf(IMFExpansionSpaceErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct IMFExpansionSpaceData));
    MFSetError(e,12,RoutineName,IMFExpansionSpaceErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return this;
   }
#endif

  data->n=n;

  MFNSpaceSetData(this,(void*)data,e);

  return this;
 }

void MFFreeExpansionSpaceData(void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"MFFreeExpansionSpaceData"};
  struct IMFExpansionSpaceData *d;

  d=(struct IMFExpansionSpaceData*)data;
  if(d!=NULL)free(d);

  return;
 }

double IMFExpansionSpaceDistance(MFNSpace this,MFNVector v0,MFNVector v1,void *d, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionSpaceDistance"};
  double result;
  int i;
  struct IMFExpansionSpaceData *data;

  data=(struct IMFExpansionSpaceData*)d;

  result=0.;
  for(i=0;i<data->n;i++)result+=pow(MFNV_C(v0,i,e)-MFNV_C(v1,i,e),2);
  result=sqrt(result);

  return result;
 }

void IMFExpansionSpaceDirection(MFNSpace this,MFNVector v0,MFNVector v1,MFNVector diff,void *d, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionSpaceDirection"};
  int i;
  struct IMFExpansionSpaceData *data;

  data=(struct IMFExpansionSpaceData*)d;

  for(i=0;i<data->n;i++)MFNVSetC(diff,i,MFNV_C(v1,i,e)-MFNV_C(v0,i,e),e);

  return;
 }

double IMFExpansionSpaceInner(MFNSpace this,MFNVector v0,MFNVector v1,void *d, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionSpaceInner"};
  double result;
  int i;
  int verbose=0;
  struct IMFExpansionSpaceData *data;

  data=(struct IMFExpansionSpaceData*)d;

#ifdef MFALLOWVERBOSE
  if(verbose)
   {
    printf("%s 0x%8.8x\n",RoutineName,this);
    printf(" v0 ");MFPrintNVector(stdout,v0,e);printf("\n");
    printf(" v1 ");MFPrintNVector(stdout,v1,e);printf("\n");
    printf(" n=%d\n",data->n);fflush(stdout);
   }
#endif

  result=0.;
  for(i=0;i<data->n;i++)
    result+=MFNV_C(v0,i,e)*MFNV_C(v1,i,e);

  return result;
 }

void IMFExpansionSpaceAdd(MFNSpace this,MFNVector v0,MFNVector v1,MFNVector sum,void *d, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionSpaceAdd"};
  int i;
  struct IMFExpansionSpaceData *data;

  data=(struct IMFExpansionSpaceData*)d;

  for(i=0;i<data->n;i++)MFNVSetC(sum,i,MFNV_C(v1,i,e)+MFNV_C(v0,i,e),e);

  return;
 }

void IMFExpansionSpaceScale(MFNSpace this,double s, MFNVector v,MFNVector prod,void *d, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionSpaceScale"};
  int i;
  struct IMFExpansionSpaceData *data;

  data=(struct IMFExpansionSpaceData*)d;

  for(i=0;i<data->n;i++)MFNVSetC(prod,i,s*MFNV_C(v,i,e),e);

  return;
 }

