/*
 *  PROGRAM NAME:  multifario
 *
 *  (c) COPYRIGHT INTERNATIONAL BUSINESS MACHINES
 *  CORPORATION 12/1/2001.  ALL RIGHTS RESERVED.
 *
 *  Please refer to the LICENSE file in the top directory
 *
 *      author: Mike Henderson mhender@watson.ibm.com
 *      date:   September 15, 2003
 */

static char *id="@(#) $Id: IMFExpansionPt.c 242 2006-08-15 21:16:26Z mhender $";

static char MFNVectorErrorMsg[256]="";

#include <MFErrorHandler.h>
#include <MFNVector.h>
#include <IMFExpansion.h>
#include <IMFExpansionSpace.h>
#include <float.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#ifndef DBL_QNAN
#define DBL_QNAN 1.e200
#endif

struct IMFExpansionNVectorData
 {
  int nC;
  int k;
  IMFExpansion E;
  double t;
  MFNVector sigma;
  MFKVector s0;
  int chart0;
  int prevChart;
  int type;
 };

static void IMFFreeExpansionNVectorData(void*,MFErrorHandler);
static int IMFExpansionNVGetNC(void*,MFErrorHandler);
static double IMFExpansionNVGetC(int,void*,MFErrorHandler);
static void IMFExpansionNVSetC(int,double,void*,MFErrorHandler);
static void IMFExpansionNVDiff(void*,void*,void*,MFErrorHandler);
static void IMFExpansionNVAdd(void*,void*,void*,MFErrorHandler);
static void IMFPrintExpansionNVector(FILE*,void*,MFErrorHandler);
static MFNVector MFCloneExpansionNVector(void*,MFErrorHandler);

/*! \fn MFNVector IMFCreateExpansionNVector(IMFExpansion E,double t, MFNVector sigma, int prevChart, int type);
 *  \brief Creates an MFNVector which has the coefficients of the expansion as coordinates. This is used for points
 *            on a fat trajectory.
 *
 *  \param E The expansion.
 *  \param t The value of the time coordinate to associate with the MFNVector (used for determined overlap).
 *  \param sigma The point (in the embedding space) of the initial "front" point to associate with the MFNVector (used for determined overlap).
 *  \param prevChart An identifying number carried with the MFNVector to indicate the preceding point on a fat trajectory.
 *  \param type Another identifier used to indicate endpoints of a fat trajectory and whether they are interpolation points
 *          or points on the manifold of initial conditions.
 */
MFNVector IMFCreateExpansionNVector(IMFExpansion E,double t, MFNVector sigma, int prevChart, int type, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFCreateExpansionNVector"};
  struct IMFExpansionNVectorData *data;
  int i;
  MFNVector this;

  data=malloc(sizeof(struct IMFExpansionNVectorData));

#ifndef MFNOSAFETYNET
  if(data==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Out of memory, trying to allocate %d bytes",sizeof(struct IMFExpansionNVectorData));
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    MFErrorHandlerOutOfMemory(e);
    return this;
   }
#endif

  data->nC=IMFExpansionN(E,e);
  data->k=IMFExpansionK(E,e);
  data->E=E;
  IMFRefExpansion(E,e);

  data->t=t;
  data->sigma=MFCloneNVector(sigma,e);
  data->s0=NULL;
  data->chart0=-1;
  data->prevChart=prevChart;
  data->type=type;

  this=MFCreateNVectorBaseClass("IMFExpansionVector",e);

  MFNVectorSetData(this,data,e);
  MFNVectorSetFreeData(this,IMFFreeExpansionNVectorData,e);
  MFNVectorSetGetNC(this,IMFExpansionNVGetNC,e);
  MFNVectorSetGetC(this,IMFExpansionNVGetC,e);
  MFNVectorSetSetC(this,IMFExpansionNVSetC,e);
  MFNVectorSetDiff(this,IMFExpansionNVDiff,e);
  MFNVectorSetAdd(this,IMFExpansionNVAdd,e);
  MFNVectorSetClone(this,MFCloneExpansionNVector,e);
  MFNVectorSetPrint(this,IMFPrintExpansionNVector,e);

  return this;
 }

void IMFFreeExpansionNVectorData(void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFFreeExpansionNVectorData"};
  struct IMFExpansionNVectorData *this;

  this=(struct IMFExpansionNVectorData*)data;

  if(this==NULL)return;
  IMFFreeExpansion(this->E,e);
  if(this->sigma!=NULL)MFFreeNVector(this->sigma,e);
  if(this->s0!=NULL)MFFreeKVector(this->s0,e);
  free(this);

  return;
 }

int IMFExpansionNVGetNC(void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFNV_NC"};
  struct IMFExpansionNVectorData *this;

  this=(struct IMFExpansionNVectorData*)data;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return -1;
   }
#endif

  return this->nC;
 }

double IMFExpansionNVGetC(int i,void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVGetC"};
  struct IMFExpansionNVectorData *this;
  double result;

  this=(struct IMFExpansionNVectorData*)data;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return DBL_QNAN;
   }

  if(i<0|| !(i<this->nC))
   {
    sprintf(MFNVectorErrorMsg,"Coordinate %d (argument 1) is illegal. Must be in 0 to %d",i,this->nC-1);
    MFSetError(e,8,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return DBL_QNAN;
   }
#endif

  result=IMFExpansionU(this->E,e)[i];

  return result;
 }

void IMFExpansionNVSetC(int i,double vl,void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVSetC"};
  struct IMFExpansionNVectorData *this;

  this=(struct IMFExpansionNVectorData*)data;

#ifdef MFNOCONFIDENCE
  if(this==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data (argument 3) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(i<0|| !(i<this->nC))
   {
    sprintf(MFNVectorErrorMsg,"Coordinate %d (argument 1) is illegal. Must be in 0 to %d",i,this->nC-1);
    MFSetError(e,8,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  IMFExpansionU(this->E,e)[i]=vl;

  return;
 }

void IMFExpansionNVDiff(void *adata,void *bdata, void *cdata, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVDiff"};
  int i,n;
  struct IMFExpansionNVectorData *a;
  struct IMFExpansionNVectorData *b;
  struct IMFExpansionNVectorData *c;

  a=(struct IMFExpansionNVectorData*)adata;
  b=(struct IMFExpansionNVectorData*)bdata;
  c=(struct IMFExpansionNVectorData*)cdata;

#ifdef MFNOCONFIDENCE
  if(a==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for a (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(b==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for b (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(c==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for c (argument 3) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(a->nC!=b->nC || a->nC!=c->nC || b->nC!=c->nC)
   {
    sprintf(MFNVectorErrorMsg,"Vectors must all be the same length a=%d, b=%d, c=%d",a->nC,b->nC,c->nC);
    MFSetError(e,4,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  n=a->nC;

  for(i=0;i<n;i++)
    IMFExpansionU(c->E,e)[i]=IMFExpansionU(a->E,e)[i]-IMFExpansionU(b->E,e)[i];

  return;
 }

void IMFExpansionNVAdd(void *adata,void *bdata,void *cdata, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVAdd"};
  int i,n;
  struct IMFExpansionNVectorData *a;
  struct IMFExpansionNVectorData *b;
  struct IMFExpansionNVectorData *c;

  a=(struct IMFExpansionNVectorData*)adata;
  b=(struct IMFExpansionNVectorData*)bdata;
  c=(struct IMFExpansionNVectorData*)cdata;

#ifdef MFNOCONFIDENCE
  if(a==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for a (argument 1) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(b==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for b (argument 2) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(c==NULL)
   {
    sprintf(MFNVectorErrorMsg,"Pointer to Vector Data for c (argument 3) is NULL");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(a->nC!=b->nC || a->nC!=c->nC || b->nC!=c->nC)
   {
    sprintf(MFNVectorErrorMsg,"Vectors must all be the same length a=%d, b=%d, c=%d",a->nC,b->nC,c->nC);
    MFSetError(e,4,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  n=a->nC;

  for(i=0;i<n;i++)
    IMFExpansionU(c->E,e)[i]=IMFExpansionU(a->E,e)[i]-IMFExpansionU(b->E,e)[i];

  return;
 }

void IMFPrintExpansionNVector(FILE *fid,void *data, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFPrintExpansionNVector"};
  int i;
  struct IMFExpansionNVectorData *u;

  u=(struct IMFExpansionNVectorData*)data;

#ifdef MFNOCONFIDENCE
  if(fid==NULL)
   {
    sprintf(MFNVectorErrorMsg,"fid (argument 1) is NULL.");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }

  if(data==NULL)
   {
    sprintf(MFNVectorErrorMsg,"data (argument 2) is NULL.");
    MFSetError(e,12,RoutineName,MFNVectorErrorMsg,__LINE__,__FILE__);
    return;
   }
#endif

  if(u->nC<1)return;

  if(u->nC<10)
   {
    fprintf(fid,"(%lf",IMFExpansionU(u->E,e)[0]);
    for(i=1;i<u->nC;i++)
      fprintf(fid,",%lf",IMFExpansionU(u->E,e)[i]);
    fprintf(fid,")");
   }else{
    fprintf(fid,"(%lf",IMFExpansionU(u->E,e)[0]);
    for(i=1;i<5;i++)
      fprintf(fid,",%lf",IMFExpansionU(u->E,e)[i]);
    fprintf(fid,",...");
    for(i=u->nC-5;i<u->nC;i++)
      fprintf(fid,",%lf",IMFExpansionU(u->E,e)[i]);
    fprintf(fid,")");
   }
  fprintf(fid,"\n");

  return;
 }

/*! \fn IMFExpansion IMFExpansionNVGetE(MFNVector this);
 *  \brief Extracts the expansion from a point on a fat trajectory.
 *
 *  \param this The MFNVector (must be an IMFExpansionNVector)
 *  \returns The expression.
 */
IMFExpansion IMFExpansionNVGetE(MFNVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVGetE"};
  struct IMFExpansionNVectorData *data;

  data=(struct IMFExpansionNVectorData*)MFNVectorGetData(this,e);

  return data->E;
 }

/*! \fn double IMFExpansionNVGetT(MFNVector this)
 *  \brief Extracts the time coordinate along the invariant manifold.
 *
 *  \param this The MFNVector (must be an MFExpansionNVector)
 *  \returns The time value.
 */
double IMFExpansionNVGetT(MFNVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVGetE"};
  struct IMFExpansionNVectorData *data;

  data=(struct IMFExpansionNVectorData*)MFNVectorGetData(this,e);

  return data->t;
 }

/*! \fn MFNVector IMFExpansionNVGetSigma(MFNVector this)
 *  \brief Extracts the sigma coordinate from an IMFExpansionNVector
 *
 *  \param this The MFNVector (must be an IMFExpansionNVector)
 *  \returns The value of sigma.
 */
MFNVector IMFExpansionNVGetSigma(MFNVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVGetSigma"};
  struct IMFExpansionNVectorData *data;

  data=(struct IMFExpansionNVectorData*)MFNVectorGetData(this,e);

  return data->sigma;
 }

MFNVector MFCloneExpansionNVector(void *d, MFErrorHandler e)
 {
  static char RoutineName[]={"MFCloneExpansionNVector"};
  struct IMFExpansionNVectorData *data;
  IMFExpansion Eclone;
  MFNVector result;
  MFKVector tmp;
  int i;

  data=(struct IMFExpansionNVectorData*)d;

  Eclone=IMFCloneExpansion(data->E,e);
  result=IMFCreateExpansionNVector(Eclone,data->t,data->sigma,data->prevChart,data->type,e);
  IMFFreeExpansion(Eclone,e);
  IMFExpansionNVSetChart0(result,data->chart0,e);
  if(data->s0!=NULL)
   {
    tmp=MFCreateKVector(MFKV_NC(data->s0,e),e);
    for(i=0;i<MFKV_NC(data->s0,e);i++)MFKVSetC(tmp,i,MFKV_C(data->s0,i,e),e);
    IMFExpansionNVSetS0(result,tmp,e);
    MFFreeKVector(tmp,e);
   }
  return result;
 }

/*! \fn int IMFExpansionNVGetType(MFNVector this);
 *  \brief Extracts the type of an IMFExpansionNVector
 *
 *  \param this The MFNVector (must be an IMFExpansionNVector)
 *  \returns The type.
 */
int IMFExpansionNVGetType(MFNVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVGetE"};
  struct IMFExpansionNVectorData *data;

  data=(struct IMFExpansionNVectorData*)MFNVectorGetData(this,e);

  return data->type;
 }

void IMFExpansionNVSetType(MFNVector this, int type, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVGetE"};
  struct IMFExpansionNVectorData *data;

  data=(struct IMFExpansionNVectorData*)MFNVectorGetData(this,e);

  data->type=type;

  return;
 }

/*! \fn MFKVector IMFExpansionNVGetS0(MFNVector this);
 *  \brief Extracts the manifold coordinates (the manifold of inital conditions) associated with the IMFExpansionNVector
 *
 *  \param this The MFNVector (must be an IMFExpansionNVector)
 *  \returns The point.
 */
MFKVector IMFExpansionNVGetS0(MFNVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVGetS0"};
  struct IMFExpansionNVectorData *data;

  data=(struct IMFExpansionNVectorData*)MFNVectorGetData(this,e);

  return data->s0;
 }

void IMFExpansionNVSetS0(MFNVector this, MFKVector S, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVSetS0"};
  struct IMFExpansionNVectorData *data;

  data=(struct IMFExpansionNVectorData*)MFNVectorGetData(this,e);

  MFRefKVector(S,e);
  if(data->s0!=NULL)MFFreeKVector(data->s0,e);
  data->s0=S;
 }

/*! \fn int IMFExpansionNVGetChart0(MFNVector this)
 *  \brief Extracts the chart (on the manifold of inital conditions) associated with the IMFExpansionNVector
 *
 *  \param this The MFNVector (must be an IMFExpansionNVector)
 *  \returns The chart number.
 */
int IMFExpansionNVGetChart0(MFNVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVGetChart0"};
  struct IMFExpansionNVectorData *data;

  data=(struct IMFExpansionNVectorData*)MFNVectorGetData(this,e);

  return data->chart0;
 }

void IMFExpansionNVSetChart0(MFNVector this, int chart0, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVSetChart0"};
  struct IMFExpansionNVectorData *data;

  data=(struct IMFExpansionNVectorData*)MFNVectorGetData(this,e);

  data->chart0=chart0;
 }

/*! \fn int IMFExpansionNVGetPrev(MFNVector this)
 *  \brief Extracts the chart that lies on the fat before the one whose center is the IMFExpansionNVector.
 *
 *  \param this The MFNVector (must be an IMFExpansionNVector)
 *  \returns The nukmber of the preceeding chart.
 */
int IMFExpansionNVGetPrev(MFNVector this, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVGetPrev"};
  struct IMFExpansionNVectorData *data;

  data=(struct IMFExpansionNVectorData*)MFNVectorGetData(this,e);

  return data->prevChart;
 }

void IMFExpansionNVSetPrev(MFNVector this, int prevChart, MFErrorHandler e)
 {
  static char RoutineName[]={"IMFExpansionNVSetPrev"};
  struct IMFExpansionNVectorData *data;

  data=(struct IMFExpansionNVectorData*)MFNVectorGetData(this,e);

  data->prevChart=prevChart;
 }
