// Copyright (c) 2004-2007 University of Geneva, HEC, Logilab
//
// OBOE is published under the Common Public License.
//
// Authors :
// Nidhi Sawhney <nsawhney@yahoo.com>
// The OBOE team
// 

#ifndef QPGENERATOR_H
#define QPGENERATOR_H

#include <iostream>
#include "AccpmGenMatrix.h"
#include "Timer.h"
#include "ExitCode.h"

#ifdef SERIALIZATION
#include <boost/archive/text_iarchive.hpp>
#include <boost/archive/text_oarchive.hpp>
#include <boost/serialization/base_object.hpp>
#endif

/**
 * @file QpGenerator.h
 * @ingroup UserInterface
 */
/**
 * The Query Point Generator class. 
 * This is the handle for the user code to iterate and get
 * query points.
 **/  

namespace Accpm {

  class Parameters;
  class Method;
  class Manager;
  class AccpmVector;
  class Oracle;
/**
 * The class responsible for generating query points.
 * It serves as the main interface to the User code, which implements
 * the Oracle functionality.
 * 
 * QpGenerator is initiliazed by providing it a parameter file, the format
 * for which is described in ProblemInput/param.oboe. It is also given
 * handle to an oracle, which is the concrete Oracle created by the user
 * for their specific application.
 *
 * Once, initialized, the quiery points are generated by subsequent calls
 * to run().
 */

  class QpGenerator {

#ifdef SERIALIZATION
    friend class boost::serialization::access;
    template<class Archive> 
      void serialize(Archive &ar, const unsigned int file_version);
#endif
    
  protected:
   Parameters *_param;
   Manager *_manager;
   Method *_method;
   Timer _timer;
   double _oracleTime;

  public:
   QpGenerator();
   virtual ~QpGenerator();
   virtual void init(Parameters *param, Oracle *oracle  = 0);
   /**
    * Each call to run() generates the next query point.
    *
    * Typically this function is put in a while loop:
    * while(run() == 0) {
    *
    * // User Code
    *
    * }
    *
    * This method internally calls the eval() function of the Oracle to 
    * compute the query point.
    * @return It can have the following return values:
    * 
    * 0 on success. 
    *
    * 1 If the number of outer iterations equals the MaxOuterIteration value.
    *
    * 2 RelativeGap has converged according to the tolerance criterion
    * controlled by parameter tolerance. 
    *
    * 3 If the user requests termination by giving a non-zero return value
    * for the eval() function.
    */
   virtual int run();
   /**
    * Clean-up and shutdown of the QpGenerator object.
   */
   virtual void terminate();
  /**
   * Display some useful statistics for the iterations at the end of the
   * run.
   */
   virtual void output(std::ostream &os);
   virtual void printHeader(std::ostream &s = std::cout) const;
   virtual void printIteration(const AccpmVector &val, const AccpmVector &val2, const AccpmGenMatrix &subGrad,
			       const AccpmGenMatrix &subProblemIndex, std::ostream &os = std::cout) const;
   
   /**
    * @return The query point at the current iteration.
    **/
   virtual const AccpmVector *getQueryPoint() const;
   
   /**
    * @return The Optimal objective value.
    * This is a valid value only at the end of successful termination of OBOE i.e QpGenerator::run() 
    * returns a status of 2.
    **/
   virtual double getOptimalObj() const;
   
   /**
    * @return The objective upper bound reached so far.
    **/
   virtual double getObjUB() const;
   
   /**
    * @return The objective lower bound reached so far.
    **/
   virtual double getObjLB() const;
  
   /**
    * @return The current relative gap which is computed as:
    * (getObjUB() - getObjLB())/|getObjUB()|;
    **/
   virtual double getRelativeGap() const;

   virtual int getNumCuts() const;
   
   /** 
    * @return The cuts(columns) stored in OBOE.
    * This is the preferred way of getting the cuts as this returns
    * a reference and does not need a copy of the cuts to be retunred.
    * This function provides all the cuts "active" internally in OBOE.
    * The cuts typically are given by the user via the Oracle::eval()
    * function, but some of the cuts might be removed internally and hence are
    * not active in OBOE.
    * One typical reason for removing a cut would be the Filter parameter, which
    * would remove a duplicate cut generated by the Oracle. There could be other
    * strategies for removing cuts. The getActiveCuts() function enables the 
    * user to get a handle to the cuts which reflect the internal cuts in OBOE.  
    **/
   virtual int getActiveCuts(const AccpmGenMatrix *&cuts) const;
   /** 
    * @return The cuts(columns) stored in OBOE
    **/
   virtual int getActiveCuts(AccpmGenMatrix &cuts) const;

   /**
    * @return The right-hand side values with respect to the
    * current cuts returned by the getActiveCuts function.
    * In other words it returns the vector c in the formulation
    * of the constraints:
    *                    Aty - Etz <= c
    * where A represents the cuts(columns).
    */

   virtual const AccpmVector& getRhs() const;
   /**
    * @return The current dual variables, x, associated
    * with the formulation.
    **/
   virtual const AccpmVector *getCurrentX() const;
   
   /**
    * @return The epigraph variable, z.
    **/
   virtual const AccpmVector& getCurrentZ() const;
   
   /**
    * @return A vector of cut indices to identify feasibility and optimality cuts.
    * The feasibility cuts have a 0 and optimality cuts have i in the vector,
    * where i is the index of the subproblem for which the cut was generated.
    * For problems with NumSubProblems = 1, it returns 1 for optimality cuts.
    * Basically, it returns the information provided by the user in the info 
    * matrix in the call to the Oracle::eval() function, the only difference 
    * is that the getCutType() function retunrs this information only for the
    * active cuts.
    * The indexing of the cut is with reference to the cuts returned by 
    * QpGenerator::getActiveCuts function.
   */
   virtual  const AccpmVector& getCutType() const;
  
   /**
    * Return the status after an iteration.
    * An exit code of value ITERATING means everything is okay and
    * more iterations need to be performed till convergence.
   **/
   virtual ExitCode getExitCode() const;
   
   /**
    * Save the state of the QpGenerator object to be able to
    * re-start in future, if desired.
    * The state is saved in fileName.
    * This requires the use of serialization library provided
    * by boost.
    * Refer --with-serialization option when configuring the system.
    * 
    * Saves the internal structures for the QpGenerator object.
    * Note that it only saves its own internal structures, the 
    * internals of the Parameters and Oracle are not saved with
    * this command.
    */
   virtual void save(const char *fileName) const;
   
   /**
    * Load the state of the QpGenerator object from fileName
    * which would have been saved by a previous call to save.
    * This requires the use of serailialization library provided
    * by boost.
    * Refer --with-serialization option when configuring the system.
    *
    * Ensure that the QpGenerator object is initialized via init() call
    * prior to laoding the internal state via the load() call.
    */
   virtual void load(const char *fileName);
  };
}

#endif
