/* $Id: log_op.hpp 1641 2010-02-01 16:39:45Z bradbell $ */
# ifndef CPPAD_LOG_OP_INCLUDED
# define CPPAD_LOG_OP_INCLUDED

/* --------------------------------------------------------------------------
CppAD: C++ Algorithmic Differentiation: Copyright (C) 2003-10 Bradley M. Bell

CppAD is distributed under multiple licenses. This distribution is under
the terms of the 
                    GNU General Public License Version 2.

A copy of this license is included in the COPYING file of this distribution.
Please visit http://www.coin-or.org/CppAD/ for information on other licenses.
-------------------------------------------------------------------------- */

CPPAD_BEGIN_NAMESPACE
/*!
\file log_op.hpp
Forward and reverse mode calculations for z = log(x).
*/

/*!
Compute forward mode Taylor coefficient for result of op = LogOp.

The C++ source code corresponding to this operation is
\verbatim
	z = log(x)
\endverbatim

\copydetails forward_unary1_op
*/
template <class Base>
inline void forward_log_op(
	size_t j           ,
	size_t i_z         ,
	size_t i_x         ,
	size_t nc_taylor   , 
	Base*  taylor      )
{	
	size_t k;

	// check assumptions
	CPPAD_ASSERT_UNKNOWN( NumArg(LogOp) == 1 );
	CPPAD_ASSERT_UNKNOWN( NumRes(LogOp) == 1 );
	CPPAD_ASSERT_UNKNOWN( i_x < i_z );
	CPPAD_ASSERT_UNKNOWN( j < nc_taylor );

	// Taylor coefficients corresponding to argument and result
	Base* x = taylor + i_x * nc_taylor;
	Base* z = taylor + i_z * nc_taylor;

	if( j == 0 )
		z[0] = log( x[0] );
	else if ( j == 1 )
		z[1] = x[1] / x[0];
	else
	{
		z[j] = -z[1] * x[j-1];
		for(k = 2; k < j; k++)
			z[j] -= Base(k) * z[k] * x[j-k];
		z[j] /= Base(j);
		z[j] += x[j];
		z[j] /= x[0];
	}
}

/*!
Compute zero order forward mode Taylor coefficient for result of op = LogOp.

The C++ source code corresponding to this operation is
\verbatim
	z = log(x)
\endverbatim

\copydetails forward_unary1_op_0
*/
template <class Base>
inline void forward_log_op_0(
	size_t i_z         ,
	size_t i_x         ,
	size_t nc_taylor   , 
	Base*  taylor      )
{

	// check assumptions
	CPPAD_ASSERT_UNKNOWN( NumArg(LogOp) == 1 );
	CPPAD_ASSERT_UNKNOWN( NumRes(LogOp) == 1 );
	CPPAD_ASSERT_UNKNOWN( i_x < i_z );
	CPPAD_ASSERT_UNKNOWN( 0 < nc_taylor );

	// Taylor coefficients corresponding to argument and result
	Base* x = taylor + i_x * nc_taylor;
	Base* z = taylor + i_z * nc_taylor;

	z[0] = log( x[0] );
}

/*!
Compute reverse mode partial derivatives for result of op = LogOp.

The C++ source code corresponding to this operation is
\verbatim
	z = log(x)
\endverbatim

\copydetails reverse_unary1_op
*/

template <class Base>
inline void reverse_log_op(
	size_t      d            ,
	size_t      i_z          ,
	size_t      i_x          ,
	size_t      nc_taylor    , 
	const Base* taylor       ,
	size_t      nc_partial   ,
	Base*       partial      )
{	size_t j, k;	

	// check assumptions
	CPPAD_ASSERT_UNKNOWN( NumArg(LogOp) == 1 );
	CPPAD_ASSERT_UNKNOWN( NumRes(LogOp) == 1 );
	CPPAD_ASSERT_UNKNOWN( i_x < i_z );
	CPPAD_ASSERT_UNKNOWN( d < nc_taylor );
	CPPAD_ASSERT_UNKNOWN( d < nc_partial );

	// Taylor coefficients and partials corresponding to argument
	const Base* x  = taylor  + i_x * nc_taylor;
	Base* px       = partial + i_x * nc_partial;

	// Taylor coefficients and partials corresponding to result
	const Base* z  = taylor  + i_z * nc_taylor;
	Base* pz       = partial + i_z * nc_partial;

	j = d;
	while(j)
	{	// scale partial w.r.t z[j]
		pz[j]   /= x[0];

		px[0]   -= pz[j] * z[j];
		px[j]   += pz[j];

		// further scale partial w.r.t. z[j]
		pz[j]   /= Base(j);

		for(k = 1; k < j; k++)
		{	pz[k]   -= pz[j] * Base(k) * x[j-k];
			px[j-k] -= pz[j] * Base(k) * z[k];
		}
		--j;
	}
	px[0] += pz[0] / x[0];
}

CPPAD_END_NAMESPACE
# endif
