#  _________________________________________________________________________
#
#  Coopr: A COmmon Optimization Python Repository
#  Copyright (c) 2008 Sandia Corporation.
#  This software is distributed under the BSD License.
#  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
#  the U.S. Government retains certain rights in this software.
#  For more information, see the Coopr README.txt file.
#  _________________________________________________________________________

__all__ = ['SystemCallSolver']

import sys
import os
from coopr.opt.base import *
from coopr.opt.base.solver import *
import tempfile

import pyutilib.common
import pyutilib.services
import pyutilib.subprocess
import pyutilib.plugin.core
import pyutilib.misc

class SystemCallSolver(OptSolver):
    """ A generic command line solver """

    def __init__(self, **kwds):
        """ Constructor """
        OptSolver.__init__(self, **kwds)
        if 'keepFiles' in kwds:
            self.keepFiles = kwds['keepFiles']
        else:
            self.keepFiles = False
        self.soln_file=None
        self.log_file=None
        self._timelimit=None
        self._timer=""

    def available(self, exception_flag=True):
        """ True if the solver is available """
        if self._assert_available:
            return True
        if not OptSolver.available(self,exception_flag):
            return False
        ans=self.executable()
        if ans is None:
           if exception_flag:
              if self.executable() is None:
                 raise pyutilib.common.ApplicationError, "No executable found for solver `"+self.name+"'"
              else:
                 raise pyutilib.common.ApplicationError, "Bad path for application `PICO' used by solver "+self.name+"': "+self.executable()
           return False
        return True

    def create_command_line(self,executable,problem_files):
        """
        Create the command line that is executed.
        """
        raise NotImplementedError       #pragma:nocover

    def process_logfile(self):
        """
        Process the logfile for information about the optimization process.
        """
        raise NotImplementedError       #pragma:nocover

    def process_other_data(self,results):
        """
        Process auxilliary data files generated by the optimizer (e.g. solution files)
        """
        pass        #pragma:nocover

    def _executable(self):
        """
        Returns the executable used by this solver.
        """
        raise NotImplementedError

    def _presolve(self, *args, **kwds):
        """
        Peform presolves.
        """
        if 'keepFiles' in kwds:
            self.keepFiles = kwds['keepFiles']
            del kwds['keepFiles']
        OptSolver._presolve(self, *args, **kwds)
        #
        # Verify that the input problem exists        
        #
        for filename in self._problem_files:
            if not os.path.exists(filename):
                raise ValueError, "Solver failed to locate input problem file="+filename
        #
        # Create command line
        #
        self._command = self.create_command_line(self.executable(),self._problem_files)
        self.log_file=self._command.log_file
        #
        # The pre-cleanup is probably unncessary, but also not harmful.
        #
        if self.log_file is not None and os.path.exists(self.log_file):
           os.remove(self.log_file)
        if self.soln_file is not None and os.path.exists(self.soln_file):
           os.remove(self.soln_file)

           
    def _apply_solver(self):
        if pyutilib.services.registered_executable("timer"):
            self._timer=pyutilib.services.registered_executable("timer").get_path()
        #
        # Execute the command
        #
        pyutilib.plugin.core.PluginGlobals.env("coopr.opt").log.debug("Running "+self._command.cmd)

        # display the log/solver file names prior to execution. this is useful
        # in case something crashes unexpectedly, which is not without precedent.
        if self.keepFiles:
           if self.log_file is not None:
              print "Solver log file=" + self.log_file
           if self.soln_file is not None:
              print "Solver solution file=" + self.soln_file
           if self._problem_files is not []:
               print "Solver problem files=",self._problem_files

        sys.stdout.flush()
        [rc,log] = self._execute_command(self._command)
        sys.stdout.flush()
        self._status = pyutilib.misc.Bunch(rc=rc, log=log)

    def _postsolve(self):

        if self.log_file is not None:
           OUTPUT=open(self.log_file,"w")
           print >>OUTPUT, "Solver command line: "+self._command.cmd
           print >>OUTPUT, ""
           print >>OUTPUT, self._status.log
           OUTPUT.close()

        # JPW: The cleanup of the problem file probably shouldn't be here, but rather
        #      in the base OptSolver class. That would require movement of the keepFiles
        #      attribute and associated cleanup logic to the base class, which I didn't
        #      feel like doing at this present time. the base class remove_files method
        #      should clean up the problem file.           

        if self.log_file is not None and not os.path.exists(self.log_file):
           raise IOError, "File " + self.log_file + " not generated while executing "+self.path
        results = None
        if self._results_format is not None:
           results = self.process_output(self._status.rc)
           if not self.keepFiles:
              pyutilib.services.TempfileManager.clear_tempfiles()
              # in some cases, the solution filename is not generated via the temp-file mechanism,
              # instead being automatically derived from the input lp/nl filename. so, we may
              # have to clean it up manually.
              if os.path.exists(self.soln_file):
                 os.remove(self.soln_file)

        return results

    def _execute_command(self,command):
        """
        Execute the command
        """
        try:
            [rc, log] = pyutilib.subprocess.run(command.cmd, timelimit=self._timelimit, env=command.env, tee=self.tee, shell=True)
        except pyutilib.common.WindowsError, err:
            raise pyutilib.common.ApplicationError, "Could not execute the command: "+command.cmd+"\tError message: "+str(err)
        sys.stdout.flush()
        return [rc,log]

    def process_output(self,rc):
        """
        Process the output files.
        """
        if self._results_format is None:
           raise ValueError, "Results format is None"
        if self.results_reader is None:
           if self._results_format is ResultsFormat.log:
              results = self.process_logfile()
           else:
              raise ValueError, "Results format `"+str(self._results_format)+"' is has not been registered"
        else:
           results = self.results_reader(self.results_file)
        self.process_other_data(results)
        if rc != None:
           results.solver.error_rc=rc
           if rc != 0:
              results.solver.status=SolverStatus.error
        return results

    def _default_results_format(self, prob_format):
        """ Returns the default results format for different problem
            formats.
        """
        return ResultsFormat.log


