#  _________________________________________________________________________
#
#  Coopr: A COmmon Optimization Python Repository
#  Copyright (c) 2008 Sandia Corporation.
#  This software is distributed under the BSD License.
#  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
#  the U.S. Government retains certain rights in this software.
#  For more information, see the Coopr README.txt file.
#  _________________________________________________________________________


import os
import os.path
import re
from mapfile_parser import parse_mapfile
import re


class MILPSymbInfo(object):

    basic_sets = ["integers","reals","literals"]

    def __init__(self,name="unknown"):
        """ Constructor. """
        self.verbose=False
        self.symbol={}
        self.superset={}
        self.basic_superset={}
        self.stype={}
        self.dimen={}
        self.slist=[]
        self.literals = []
        self.name=name
        #
        # A regular expression used to match valid set names
        self.re_word = re.compile('[a-zA-Z][a-zA-Z_0-9\.]*$')
        #
        # temp_sets[setname] is the name of a temporary set for 'setname''
        self.temp_sets = {}
        #
        # orig_sets[setname] is the name of the original set for temporary set 'setname'
        self.orig_sets = {}
        #
        # The index of the next valid temporary set (set#)
        self.tempid=0

    def read_mapfile(self,filename=None,debug=0):
        """ Read a mapfile and initialize this class """
        if filename is None:
            filename = self.name+".map"
        if not os.path.exists(filename):
            raise IOError, "File does not exist: "+filename
        #
        # Call PLY parser
        #
        parse_mapfile(self, filename=filename, debug=debug)

    def write_mapfile(self, filename=None,quiet=False):
        """ Write a mapfile from the data in this class """
        if filename is None:
            filename = self.name+".map"
        OUTPUT = open(filename,"w")
        basename = os.path.basename(filename)
        print >>OUTPUT, "#"
        print >>OUTPUT, "# Mapfile "+basename+" generated by the"
        print >>OUTPUT, "#    SUCASA MILPSymbInfo class."
        print >>OUTPUT, "#"
        literal_superset = []
        unknown_dimen = []
        for name in self.slist:
            tmp = self.stype[name]+" "+name
            if len(self.symbol[name])>0:
               tmp += "["
               flag=False
               for val in self.symbol[name]:
                    if flag:
                        tmp += ","
                    flag=True
                    tmp += val
               tmp += "]"
            if self.stype[name] == "set": 
                   if type(self.superset[name]) in (list,tuple):
                        tmp += " within "+self.superset[name][0]
                        for superset in self.superset[name][1:]:
                            tmp += " cross "+superset
                   else:
                        tmp += " within "+str(self.superset[name])
                   if self.dimen[name] == 0:
                        unknown_dimen.append(name)
                        tmp += " dimen 1"
                   else:
                        tmp += " dimen "+str(self.dimen[name])
            elif self.stype[name] == "param":
                if type(self.superset[name]) is list:
                    if len(self.superset[name]) == 1:
                        tmp += " in "+self.superset[name][0]
                    else:
                        tmp += " in "+self.superset[name][0]
                        for i in range(1,len(self.superset[name])):
                            tmp += " cross "+self.superset[name][i]
                else:
                    tmp += " in "+self.superset[name]
            if self.superset[name] == "literals":
                literal_superset.append(name)
            tmp += ";"
            print >>OUTPUT, tmp
        OUTPUT.close()
        if not quiet and len(literal_superset) > 0:     #pragma:nocover
            print "      WARNING: superset type unknown for the following symbols, so"
            print "        they are interpreted as symbolic literals:"
            for id in literal_superset:
                print "         ",id
        if not quiet and len(unknown_dimen) > 0:        #pragma:nocover
            print "      WARNING: dimen unknown for the following symbols, so"
            print "        they are assumed to have dimen 1:"
            for id in unknown_dimen:
                print "         ",id
        return filename

    def __str__(self):
        """ Generate a string representation """
        ans = ""
        for name in self.slist:
            ans += str(self.stype[name]) + " " + name + " " +  str(self.symbol[name])+" "+str(self.superset[name])+" "+str(self.dimen[name])+"\n"
        return ans
        
    def print_symbols(self):
        """ Print the string representation """
        print str(self)

    def add_symbol(self,stype,name,index=None,superset=None,tmpsets=False,dimen=1,quiet=False):
        """ 
        Add a symbol.
        stype - the type of the symbol
        name - the name of the symbol
        index - a tuple of index sets
        dimen - the dimension of the data that this symbol points to
        """
        if name in self.symbol:
            raise IOError, "Symbol '"+name+"' has already been defined!"
        if not index is None:
            if tmpsets:
                #
                # We create temporary names for sets that aren't already
                # specified
                #
                tmp=[]
                for id in index:
                    if not id in self.basic_sets and not id in self.symbol:
                        if re.match(self.re_word,id):
                            self.add_symbol("set",id)
                            tmp.append(id)
                        elif id in self.temp_sets:
                            tmp.append(self.temp_sets[id])
                        else:
                            newname = "set"+str(self.tempid)
                            if not quiet:           #pragma:nocover
                                print "      WARNING: creating set '"+newname+"' for set string '"+id+"'"
                            self.add_symbol("set",newname)
                            tmp.append(newname)
                            self.temp_sets[id]=newname
                            self.orig_sets[newname]=id
                            self.tempid += 1
                    else:
                        tmp.append(id)
                index=tmp
            else:
                #
                # We verify that all set names have been specified
                #
                for id in index:
                    if not id in self.basic_sets and not id in self.symbol:
                        raise IOError, "Bad index set name '"+id+"' for symbol '"+name+"'"
        if not superset is None:
            if not type(superset) is list:
                if not superset in self.basic_sets and not superset in self.symbol:
                    raise IOError, "Bad superset set name '"+superset+"' for symbol "+name
            else:
                for supersetitem in superset:
                    if not supersetitem in self.basic_sets and not supersetitem in self.symbol:
                        raise IOError, "Bad superset set name '"+supersetitem+"' for symbol "+name
            
        #
        # SUCASA defaults
        #
        if superset is None and stype == "set":
            superset="literals"
        elif superset is None and stype == "param":
            superset="reals"

        if superset is not None:
            if type(superset) in [list,tuple]:
                key = tuple(superset)
            else:
                key = (superset,)
            if key in self.basic_superset:
                self.basic_superset[(name,)]= self.basic_superset[key]
            else:
                self.basic_superset[(name,)] = key
        if index is None:
            self.symbol[name]=[]
        else:
            self.symbol[name]=index
        self.superset[name]=superset
        self.stype[name]=stype
        self.dimen[name]=int(dimen)
        self.slist.append(name)

    def generate_milp_symbol_code(self):
        """ Generate the MILP code """
        #
        # Write Info Header
        #
        OUTPUT = open(self.name+"_info.h","w")
        print >>OUTPUT, "//"
        print >>OUTPUT, "// " + self.name + "_info.h"
        print >>OUTPUT, "//"
        print >>OUTPUT, "// Automatically generated by MILPSymbInfo::setup_ilp_symbols"
        print >>OUTPUT, ""
        print >>OUTPUT, "//"
        print >>OUTPUT, ""
        print >>OUTPUT, "#ifndef " + self.name + "_info_h"
        print >>OUTPUT, "#define " + self.name + "_info_h"
        print >>OUTPUT, ""
        print >>OUTPUT, "#include <set>"
        print >>OUTPUT, "#include <map>"
        print >>OUTPUT, "#include <pico/MILPSymbInfo.h>"
        print >>OUTPUT, "#include <pico/MILPSymbol.h>"
        print >>OUTPUT, "#include <utilib/BasicArray.h>"
        print >>OUTPUT, ""
        print >>OUTPUT, "namespace sucasa_" + self.name + " {"
        print >>OUTPUT, ""
        print >>OUTPUT, "using utilib::CharString;"
        print >>OUTPUT, "using pico::MILPSymbol0;" 
        print >>OUTPUT, "using pico::MILPSymbol1;" 
        print >>OUTPUT, "using pico::MILPSymbol2;" 
        print >>OUTPUT, "using pico::MILPSymbol3;" 
        print >>OUTPUT, "//using pico::MILPSymbol4;" 
        print >>OUTPUT, "//using pico::MILPSymbol5;" 
        print >>OUTPUT, "//using pico::MILPSymbol6;" 
        print >>OUTPUT, "//using pico::MILPSymbol7;" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "class Info;" 
        print >>OUTPUT, "" 

        print >>OUTPUT, "" 
        #
        # Define SymbInfo class
        #
        print >>OUTPUT, "///" 
        print >>OUTPUT, "class SymbInfo" 
        print >>OUTPUT, "{" 
        print >>OUTPUT, "public:" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  typedef pico::MILPSymbol::tuple_t tuple_t;" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  SymbInfo(Info* _info=0) : infoptr(_info) {}" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  Info* infoptr;" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  Info& info() {return *infoptr;}" 
        print >>OUTPUT, "};" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "" 

        print >>OUTPUT, "" 
        #
        # Define MILPSymbFunctions class
        #
        print >>OUTPUT, "class MILPSymbFunctions : public SymbInfo"
        print >>OUTPUT, "{" 
        print >>OUTPUT, "public:" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  MILPSymbFunctions(Info* _info=NULL) : SymbInfo(_info) {}" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  void test_varmap(utilib::BasicArray<double>& primal, utilib::BasicArray<double>& dual);" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  void register_var_values(utilib::BasicArray<double>& vec);" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  void register_dual_values(utilib::BasicArray<double>& vec);" 
        print >>OUTPUT, "  ///" 
        #
        # typedefs for symbol iterators
        #
        print >>OUTPUT, "  typedef std::set<SymbInfo::tuple_t >::iterator symbol_iterator;"
        print >>OUTPUT, "  typedef std::set<SymbInfo::tuple_t >::const_iterator symbol_const_iterator;"
        print >>OUTPUT, "  typedef std::set<size_t>::iterator set_iterator;"
        print >>OUTPUT, "  typedef std::set<size_t>::const_iterator set_const_iterator;"
        print >>OUTPUT, ""
        #
        # Print symbol functions 
        #
        for name in self.symbol:
            type = self.stype[name]
            #
            # CharString& foo_name()
            #
            print >>OUTPUT, "  const CharString& "+name+"_name() const;"
            #
            # size_t foo(int& arg1, double& arg2) const;
            # size_t bar() const;
            #
            if type == "set":
                if self.dimen[name] <= 1:
                    print >>OUTPUT, "  const std::set<size_t>& "+name+"("+self._set_args(self.symbol[name])+") const;"
                else:
                    print >>OUTPUT, "  const std::set<SymbInfo::tuple_t>& "+name+"("+self._set_args(self.symbol[name])+") const;    // dimen="+str(self.dimen[name])
            elif type == "param":
               print >>OUTPUT, "  const "+self._index_str(name)+"& "+name+"("+self._set_args(self.symbol[name])+") const;"
               print >>OUTPUT, "  const "+self._index_str(name)+"& "+name+"_value("+self._set_args(self.symbol[name])+") const;"
               print >>OUTPUT, "  size_t "+name+"_index("+self._set_args(self.symbol[name])+") const;"
            elif type == "var":
               print >>OUTPUT, "  double "+name+"("+self._set_args(self.symbol[name])+") const;"
               print >>OUTPUT, "  double "+name+"_value("+self._set_args(self.symbol[name])+") const;"
               print >>OUTPUT, "  size_t "+name+"_index("+self._set_args(self.symbol[name])+") const;"
            else:
               print >>OUTPUT, "  size_t "+name+"("+self._set_args(self.symbol[name])+") const;"
               print >>OUTPUT, "  double "+name+"_value("+self._set_args(self.symbol[name])+") const;"
               print >>OUTPUT, "  size_t "+name+"_index("+self._set_args(self.symbol[name])+") const;"
            #
            # const std::string& foo_value(size_t)
            # const double& foo_value(size_t)
            #
            if (type == "set" and self.dimen[name] <= 1):
                print >>OUTPUT, "  const "+self._index_str(name)+"& "+name+"_value(const size_t& idx) const;"
            if type == "set" and self.dimen[name] > 1:
                print >>OUTPUT, "  utilib::Any "+name+"_value(const SymbInfo::tuple_t& tuple) const;"
            #
            # Functions for indexed data
            #
            if len(self.symbol[name]) > 0:
                #
                # size_t foo(const SymbInfo::tuple_t& tuple) const;
                #
                if type == "set":
                    if self.dimen[name] <= 1:
                        print >>OUTPUT, "  const std::set<size_t>& "+name+"(const SymbInfo::tuple_t& tuple) const;"
                    else:
                        print >>OUTPUT, "  const std::set<SymbInfo::tuple_t>& "+name+"(const SymbInfo::tuple_t& tuple) const;"
                elif type == "param":
                    print >>OUTPUT, "  const "+self._index_str(name)+"& "+name+"(const SymbInfo::tuple_t& tuple) const;"
                    print >>OUTPUT, "  const "+self._index_str(name)+"& "+name+"_value(const SymbInfo::tuple_t& tuple) const;"
                    print >>OUTPUT, "  size_t "+name+"_index(const SymbInfo::tuple_t& tuple) const;"
                elif type == "var":
                    print >>OUTPUT, "  double "+name+"(const SymbInfo::tuple_t& tuple) const;"
                    print >>OUTPUT, "  double "+name+"_value(const SymbInfo::tuple_t& tuple) const;"
                    print >>OUTPUT, "  size_t "+name+"_index(const SymbInfo::tuple_t& tuple) const;"
                else:
                    print >>OUTPUT, "  size_t "+name+"(const SymbInfo::tuple_t& tuple) const;"
                    print >>OUTPUT, "  double "+name+"_value(const SymbInfo::tuple_t& tuple) const;"
                    print >>OUTPUT, "  size_t "+name+"_index(const SymbInfo::tuple_t& tuple) const;"
                #
                # bool foo_isvalid(const SymbInfo::tuple_t& tuple) const;
                #
                print >>OUTPUT, "  bool "+name+"_isvalid(const SymbInfo::tuple_t& tuple) const;"

                #
                # bool foo_isvalid(size_t& arg1, size_t& arg2) const;
                #
                print >>OUTPUT, "  bool "+name+"_isvalid("+self._set_args(self.symbol[name])+") const;"

                #
                # const set<SymbInfo::tuple_t >& foo_indices() const;
                #
                print >>OUTPUT, "  const std::set<SymbInfo::tuple_t >& "+name+"_indices() const;"

            print >>OUTPUT, ""
        print >>OUTPUT, "};" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "" 
        #
        # Define Info class
        #
        print >>OUTPUT, "class Info : public pico::MILPSymbInfo, public MILPSymbFunctions"
        print >>OUTPUT, "{" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "public:" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  Info();" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  ~Info() {}" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  utilib::BasicArray<double> primal_values;" 
        print >>OUTPUT, "  ///" 
        print >>OUTPUT, "  utilib::BasicArray<double> dual_values;" 

        for name in self.symbol:
            print >>OUTPUT, "  ///" 
            if self.stype[name] == "set":
                if self.dimen[name] <= 1:
                   print >>OUTPUT, "  MILPSymbol"+str(len(self.symbol[name]))+"<"+self._index_str(name)+",size_t> "+name+"_symbol;"
                   print >>OUTPUT, "  typedef MILPSymbol"+str(len(self.symbol[name]))+"<"+self._index_str(name)+",size_t>::item_iterator "+name+"_iterator;"
                   print >>OUTPUT, "  typedef MILPSymbol"+str(len(self.symbol[name]))+"<"+self._index_str(name)+",size_t>::item_const_iterator "+name+"_const_iterator;"
                else:
                   print >>OUTPUT, "  MILPSymbol"+str(len(self.symbol[name]))+"<void,SymbInfo::tuple_t> "+name+"_symbol;"
                   print >>OUTPUT, "  typedef MILPSymbol"+str(len(self.symbol[name]))+"<void,SymbInfo::tuple_t>::item_iterator "+name+"_iterator;"
                   print >>OUTPUT, "  typedef MILPSymbol"+str(len(self.symbol[name]))+"<void,SymbInfo::tuple_t>::item_const_iterator "+name+"_const_iterator;"
            else:
                print >>OUTPUT, "  MILPSymbol"+str(len(self.symbol[name]))+"<"+self._index_str(name)+",size_t,size_t> "+name+"_symbol;"
                print >>OUTPUT, "  typedef MILPSymbol"+str(len(self.symbol[name]))+"<"+self._index_str(name)+",size_t,size_t>::item_iterator "+name+"_iterator;"
                print >>OUTPUT, "  typedef MILPSymbol"+str(len(self.symbol[name]))+"<"+self._index_str(name)+",size_t,size_t>::item_const_iterator "+name+"_const_iterator;"
            print >>OUTPUT, ""
        print >>OUTPUT, "};"
        print >>OUTPUT, ""
        print >>OUTPUT, ""
        print >>OUTPUT, ""
        print >>OUTPUT, "";
        #
        # Print methods for the MILPSymbFunctions class
        #
        for name in self.symbol:
            type = self.stype[name]
            #
            # CharString& foo_name()
            #
            print >>OUTPUT, "  inline const CharString& MILPSymbFunctions::"+name+"_name() const"
            print >>OUTPUT, "    {return infoptr->"+name+"_symbol.name;}"
            #
            # std::set<size_t> foo()
            #   OR
            # size_t foo()
            #
            if type == "set":
                if self.dimen[name] <= 1:
                    print >>OUTPUT, "  inline const std::set<size_t>& MILPSymbFunctions::"+name+"("+self._set_args(self.symbol[name])+") const"
                else:
                    print >>OUTPUT, "  inline const std::set<SymbInfo::tuple_t>& MILPSymbFunctions::"+name+"("+self._set_args(self.symbol[name])+") const"
                print >>OUTPUT, "    {return infoptr->"+name+"_symbol("+self._args(self.symbol[name])+");}"
            elif type == "param":
                print >>OUTPUT, "  inline const "+self._index_str(name)+"& MILPSymbFunctions::"+name+"("+self._set_args(self.symbol[name])+") const"
                print >>OUTPUT, "    {return "+name+"_value("+self._args(self.symbol[name])+");}"
                print >>OUTPUT, "  inline const "+self._index_str(name)+"& MILPSymbFunctions::"+name+"_value("+self._set_args(self.symbol[name])+") const"
                if (name,) not in self.basic_superset:          #pragma:nocover
                    raise IOError, "Attempting to setup '"+name+"' literal value but it is not have a basic superset"
                elif self.basic_superset[(name,)] == ("integers",):
                    print >>OUTPUT, "    {return infoptr->"+name+"_symbol("+self._args(self.symbol[name])+");}"
                elif self.basic_superset[(name,)] == ("reals",):
                    print >>OUTPUT, "    {return infoptr->double_literals[infoptr->"+name+"_symbol("+self._args(self.symbol[name])+")];}"
                elif self.basic_superset[(name,)] == ("literals",):
                    print >>OUTPUT, "    {return infoptr->string_literals[infoptr->"+name+"_symbol("+self._args(self.symbol[name])+")];}"
                print >>OUTPUT, "  inline size_t MILPSymbFunctions::"+name+"_index("+self._set_args(self.symbol[name])+") const"
                print >>OUTPUT, "    {return infoptr->"+name+"_symbol("+self._args(self.symbol[name])+");}"
            elif type == "var":
               print >>OUTPUT, "  inline double MILPSymbFunctions::"+name+"("+self._set_args(self.symbol[name])+") const"
               print >>OUTPUT, "    {return infoptr->primal_values[infoptr->"+name+"_symbol("+self._args(self.symbol[name])+")];}"
               print >>OUTPUT, "  inline double MILPSymbFunctions::"+name+"_value("+self._set_args(self.symbol[name])+") const"
               print >>OUTPUT, "    {return infoptr->primal_values[infoptr->"+name+"_symbol("+self._args(self.symbol[name])+")];}"
               print >>OUTPUT, "  inline size_t MILPSymbFunctions::"+name+"_index("+self._set_args(self.symbol[name])+") const"
               print >>OUTPUT, "    {return infoptr->"+name+"_symbol("+self._args(self.symbol[name])+");}"
            else:
               print >>OUTPUT, "  inline size_t MILPSymbFunctions::"+name+"("+self._set_args(self.symbol[name])+") const"
               print >>OUTPUT, "    {return infoptr->"+name+"_symbol("+self._args(self.symbol[name])+");}"
               print >>OUTPUT, "  inline double MILPSymbFunctions::"+name+"_value("+self._set_args(self.symbol[name])+") const"
               print >>OUTPUT, "    {return infoptr->dual_values[infoptr->"+name+"_symbol("+self._args(self.symbol[name])+")];}"
               print >>OUTPUT, "  inline size_t MILPSymbFunctions::"+name+"_index("+self._set_args(self.symbol[name])+") const"
               print >>OUTPUT, "    {return infoptr->"+name+"_symbol("+self._args(self.symbol[name])+");}"
            #
            # const std::string& foo_value(size_t)
            # const double& foo_value(size_t)
            #
            if (type == "set" and self.dimen[name] <= 1):
                print >>OUTPUT, "  inline const "+self._index_str(name)+"& MILPSymbFunctions::"+name+"_value(const size_t& idx) const"
                if (name,) not in self.basic_superset:          #pragma:nocover
                    raise IOError, "Attempting to setup '"+name+"' literal value but it is not have a basic superset"
                elif self.basic_superset[(name,)] == ("integers",):
                    print >>OUTPUT, "    {return idx;}"
                elif self.basic_superset[(name,)] == ("reals",):
                    print >>OUTPUT, "    {return infoptr->double_literals[idx];}"
                elif self.basic_superset[(name,)] == ("literals",):
                    print >>OUTPUT, "    {return infoptr->string_literals[idx];}"
            if type == "set" and self.dimen[name] > 1:
                print >>OUTPUT, " inline utilib::Any MILPSymbFunctions::"+name+"_value(const SymbInfo::tuple_t& tuple) const"
                print >>OUTPUT, "   { return infoptr->"+name+"_symbol.value(tuple);}"
            #
            # Functions for indexed data
            #
            if len(self.symbol[name]) > 0:
                #
                # bool foo_isvalid(const SymbInfo::tuple_t& tuple) const;
                #
                print >>OUTPUT, "  inline bool MILPSymbFunctions::"+name+"_isvalid(const SymbInfo::tuple_t& tuple) const"
                print >>OUTPUT, "    { try { infoptr->"+name+"_symbol(tuple);} catch (std::runtime_error& err) {return false;} return true; }"
                #
                # bool foo_isvalid(int& arg1, double& arg2) const;
                #
                print >>OUTPUT, "  inline bool MILPSymbFunctions::"+name+"_isvalid("+self._set_args(self.symbol[name])+") const"
                print >>OUTPUT, "    { try { infoptr->"+name+"_symbol("+self._args(self.symbol[name])+");} catch (std::runtime_error& err) {return false;} return true;}"
                #
                # std::set<SymbInfo::tuple_t >& foo_indices() const;
                #
                print >>OUTPUT, "  inline const std::set<SymbInfo::tuple_t >& MILPSymbFunctions::"+name+"_indices() const"
                print >>OUTPUT, "    {return infoptr->"+name+"_symbol.valid;}"
                #
                # int foo(const SymbInfo::tuple_t& tuple) const;
                #
                if type == "set":
                    if self.dimen[name] <= 1:
                        print >>OUTPUT, "  inline const std::set<size_t>& MILPSymbFunctions::"+name+"(const SymbInfo::tuple_t& tuple) const"
                    else:
                        print >>OUTPUT, "  inline const std::set<SymbInfo::tuple_t>& MILPSymbFunctions::"+name+"(const SymbInfo::tuple_t& tuple) const"
                    print >>OUTPUT, "    {return infoptr->"+name+"_symbol(tuple);}"
                elif type == "param":
                    print >>OUTPUT, "  inline const "+self._index_str(name)+"& MILPSymbFunctions::"+name+"(const SymbInfo::tuple_t& tuple) const"
                    print >>OUTPUT, "    {return "+name+"_value(tuple);}"
                    print >>OUTPUT, "  inline const "+self._index_str(name)+"& MILPSymbFunctions::"+name+"_value(const SymbInfo::tuple_t& tuple) const"
                    if (name,) not in self.basic_superset:          #pragma:nocover
                        raise IOError, "Attempting to setup '"+name+"' literal value but it is not have a basic superset"
                    elif self.basic_superset[(name,)] == ("integers",):
                        print >>OUTPUT, "    {return infoptr->"+name+"_symbol(tuple);}"
                    elif self.basic_superset[(name,)] == ("reals",):
                        print >>OUTPUT, "    {return infoptr->double_literals[infoptr->"+name+"_symbol(tuple)];}"
                    elif self.basic_superset[(name,)] == ("literals",):
                        print >>OUTPUT, "    {return infoptr->string_literals[infoptr->"+name+"_symbol(tuple)];}"

                    print >>OUTPUT, "  inline size_t MILPSymbFunctions::"+name+"_index(const SymbInfo::tuple_t& tuple) const"
                    print >>OUTPUT, "    {return infoptr->"+name+"_symbol(tuple);}"
                elif type == "var":
                    print >>OUTPUT, "  inline double MILPSymbFunctions::"+name+"(const SymbInfo::tuple_t& tuple) const"
                    print >>OUTPUT, "    {return infoptr->primal_values[infoptr->"+name+"_symbol(tuple)];}"
                    print >>OUTPUT, "  inline double MILPSymbFunctions::"+name+"_value(const SymbInfo::tuple_t& tuple) const"
                    print >>OUTPUT, "    {return infoptr->primal_values[infoptr->"+name+"_symbol(tuple)];}"
                    print >>OUTPUT, "  inline size_t MILPSymbFunctions::"+name+"_index(const SymbInfo::tuple_t& tuple) const"
                    print >>OUTPUT, "    {return infoptr->"+name+"_symbol(tuple);}"
                else:
                    print >>OUTPUT, "  inline size_t MILPSymbFunctions::"+name+"(const SymbInfo::tuple_t& tuple) const"
                    print >>OUTPUT, "    {return infoptr->"+name+"_symbol(tuple);}"
                    print >>OUTPUT, "  inline double MILPSymbFunctions::"+name+"_value(const SymbInfo::tuple_t& tuple) const"
                    print >>OUTPUT, "    {return infoptr->dual_values[infoptr->"+name+"_symbol(tuple)];}"
                    print >>OUTPUT, "  inline size_t MILPSymbFunctions::"+name+"_index(const SymbInfo::tuple_t& tuple) const"
                    print >>OUTPUT, "    {return infoptr->"+name+"_symbol(tuple);}"
            print >>OUTPUT, ""

        print >>OUTPUT, "" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "}"
        print >>OUTPUT, ""
                

        print >>OUTPUT, "" 
        print >>OUTPUT, "#endif" 
        OUTPUT.close()

        #
        # Write Info Source
        #
        OUTPUT = open(self.name+"_info.cpp","w")
        print >>OUTPUT, "//"
        print >>OUTPUT, "// " + self.name + "_info.cpp"
        print >>OUTPUT, "// Automatically generated by MILPSymbInfo::setup_ilp_symbols"
        print >>OUTPUT, "//"
        print >>OUTPUT, "" 
        print >>OUTPUT, "#include <utilib/CommonIO.h>" 
        print >>OUTPUT, "#include \"" + self.name + "_info.h\"" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "namespace sucasa_" + self.name + " {" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "using namespace std;" 
        print >>OUTPUT, "using namespace pico;" 
        print >>OUTPUT, "using namespace utilib;" 
        print >>OUTPUT, "" 

        print >>OUTPUT, "Info::Info()" 
        print >>OUTPUT, "  : MILPSymbFunctions(this)" 
        for name in self.symbol:
            if self.stype[name] == "set":
                print >>OUTPUT, "    ,"+name+"_symbol(\""+name+"\")"
            elif self.superset[name] == "reals":
               print >>OUTPUT, "    ,"+name+"_symbol(\""+name+"\",&double_literals)"
            elif self.superset[name] == "literals":
               print >>OUTPUT, "    ,"+name+"_symbol(\""+name+"\",&string_literals)"
            else:
               print >>OUTPUT, "    ,"+name+"_symbol(\""+name+"\")"
        print >>OUTPUT, "  {" 
        for name in self.symbol:
            print >>OUTPUT, "    //"
            print >>OUTPUT, "    // Initialize the "+name+"_symbol object"
            print >>OUTPUT, "    //"
            print >>OUTPUT, "    "+name+"_symbol.dimen = "+str(max(1,self.dimen[name]))+" ;"
            print >>OUTPUT, "    symbols[\""+name+"\"] = &"+name+"_symbol;"
            if not self.stype[name] in ["set","param"]:
                print >>OUTPUT, "    rc_symbols[\""+name+"\"] = &"+name+"_symbol;"
            for arg in self.symbol[name]:
               print >>OUTPUT, "    "+name+"_symbol.arg_sets.push_back(&"+arg+"_symbol);"
            if self.stype[name] == "set" and self.dimen[name] <= 1:
                if self.basic_superset[(name,)] == ("literals",):
                   print >>OUTPUT, "    "+name+"_symbol.set_literals(&string_literals);"
                elif self.basic_superset[(name,)] == ("reals",):
                   print >>OUTPUT, "    "+name+"_symbol.set_literals(&double_literals);"
            print >>OUTPUT, ""

        print >>OUTPUT, "  }" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "" 
        
        print >>OUTPUT, "void MILPSymbFunctions::register_var_values(utilib::BasicArray<double>& vec)" 
        print >>OUTPUT, "{ infoptr->primal_values &= vec; }" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "" 

        print >>OUTPUT, "void MILPSymbFunctions::register_dual_values(utilib::BasicArray<double>& vec)" 
        print >>OUTPUT, "{ infoptr->dual_values &= vec; }" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "" 

        print >>OUTPUT, "void MILPSymbFunctions::test_varmap(utilib::BasicArray<double>& primal, utilib::BasicArray<double>& dual)" 
        print >>OUTPUT, "{" 
        print >>OUTPUT, "register_var_values(primal);" 
        print >>OUTPUT, "register_dual_values(dual);" 
        print >>OUTPUT, ""
        print >>OUTPUT, "ucout << \" ---- BEGINNING test_varmap OUTPUT ----------------------------\" << std::endl;"
        for name in self.symbol:
            indices = "(*curr)[0]"
            for i in range(1,len(self.symbol[name])):
                indices = indices + ",(*curr)[" + str(i) +"]"
            if self.stype[name] == "set":
                print >>OUTPUT, "ucout << \"Set " +name+"\" << std::endl;" 
            else:
                print >>OUTPUT, "ucout << \"Symbol " +name+"\" << std::endl;" 
            print >>OUTPUT, "ucout << \""+name+" name: \" << " +name+"_name() << std::endl;" 
            #print "HERE",name,self.symbol[name], self.dimen[name]
            if len(self.symbol[name]) > 0:

                if self.stype[name] == "set":
                   print >>OUTPUT, "{" 
                   print >>OUTPUT, "symbol_const_iterator curr = " + name +"_indices().begin();" 
                   print >>OUTPUT, "symbol_const_iterator last = " + name + "_indices().end();" 
                   print >>OUTPUT, "ucout << \"tuple: \" << *curr << std::endl;"
                   print >>OUTPUT, "while (curr != last) {" 
                   print >>OUTPUT, "  {ucout << \""+name+" value: \";"
                   print >>OUTPUT, "  set_const_iterator scurr = " +name+"(*curr).begin();"
                   print >>OUTPUT, "  set_const_iterator send = " +name+"(*curr).end();"
                   print >>OUTPUT, "  while (scurr != send) {"
                   print >>OUTPUT, "    ucout << " +name+"_value(*scurr);"
                   print >>OUTPUT, "    ucout << \" ( \" << *scurr << \" ) \";"
                   print >>OUTPUT, "    scurr++;"
                   print >>OUTPUT, "    }"
                   print >>OUTPUT, "  ucout << std::endl;}"
                   print >>OUTPUT, "  {ucout << \""+name+" value: \";"
                   print >>OUTPUT, "  set_const_iterator scurr = " +name+"("+indices+").begin();"
                   print >>OUTPUT, "  set_const_iterator send = " +name+"("+indices+").end();"
                   print >>OUTPUT, "  while (scurr != send) {"
                   print >>OUTPUT, "    ucout << " +name+"_value(*scurr);"
                   print >>OUTPUT, "    ucout << \" ( \" << *scurr << \" ) \";"
                   print >>OUTPUT, "    scurr++;"
                   print >>OUTPUT, "    }"
                   print >>OUTPUT, "  ucout << std::endl;}"
                   print >>OUTPUT, "  ucout << \"isvalid: \" << "+name+"_isvalid(*curr) << std::endl;"
                   print >>OUTPUT, "  ucout << \"isvalid: \" << "+name+"_isvalid("+indices+") << std::endl;"
                else:
                   print >>OUTPUT, "{" 
                   print >>OUTPUT, "symbol_const_iterator curr = " + name +"_indices().begin();" 
                   print >>OUTPUT, "symbol_const_iterator last = " + name + "_indices().end();" 
                   print >>OUTPUT, "while (curr != last) {" 
                   print >>OUTPUT, "  ucout << \""+name+" \" << "+name+"(*curr) << std::endl;"
                   print >>OUTPUT, "  ucout << \""+name+" \" << "+name+"("+indices+") << std::endl;"
                   print >>OUTPUT, "  ucout << \""+name+" value: \" << "+name+"_value(*curr) << std::endl;"
                   print >>OUTPUT, "  ucout << \""+name+" value: \" << "+name+"_value("+indices+") << std::endl;"
                   print >>OUTPUT, "  ucout << \""+name+" index: \" << "+name+"_index(*curr) << std::endl;"
                   print >>OUTPUT, "  ucout << \""+name+" index: \" << "+name+"_index("+indices+") << std::endl;"
                   print >>OUTPUT, "  ucout << \"tuple: \" << *curr << std::endl;"
                   print >>OUTPUT, "  ucout << \"isvalid: \" << "+name+"_isvalid(*curr) << std::endl;"
                   print >>OUTPUT, "  ucout << \"isvalid: \" << "+name+"_isvalid("+indices+") << std::endl;"
                print >>OUTPUT, "  curr++;" 
                print >>OUTPUT, "  }" 
                print >>OUTPUT, "}" 
            else:
                if self.stype[name] == "set":
                    print >>OUTPUT, "{"
                    print >>OUTPUT, "set_const_iterator curr = "+name+"().begin();"
                    print >>OUTPUT, "set_const_iterator end  = "+name+"().end();"
                    print >>OUTPUT, "while (curr != end) {"
                    print >>OUTPUT, "  ucout << \""+name+" value: \" << "+name+"_value(*curr) << \"  "+name+" index: \" << *curr << std::endl;"
                    print >>OUTPUT, "  curr++;"
                    print >>OUTPUT, "  }"
                    print >>OUTPUT, "}"
                else:
                    print >>OUTPUT, "ucout << \""+name+": \" << " +name+"() << std::endl;" 
                    print >>OUTPUT, "ucout << \""+name+" value: \" << " +name+"_value() << std::endl;" 
                    print >>OUTPUT, "ucout << \""+name+" index: \" << " +name+"_index() << std::endl;" 
            print >>OUTPUT, "ucout << std::endl;" 

        print >>OUTPUT, "ucout << \" ---- ENDING test_varmap OUTPUT ----------------------------\" << std::endl;"
        print >>OUTPUT, "ucout << Flush;" 
        print >>OUTPUT, "}" 
        print >>OUTPUT, "" 
        print >>OUTPUT, "} // namespace sucasa_" + self.name 
        print >>OUTPUT, "" 

        OUTPUT.close()
        return [self.name+"_info.h",self.name+"_info.cpp"]



    def _index_str(self,name):
        # WEH - I don't think that this is needed
        #if name in self.basic_sets:
            #return name
        if (name,) not in self.basic_superset:
            return "size_t"
        if self.basic_superset[(name,)] == ("integers",):
            return "size_t"
        elif self.basic_superset[(name,)] == ("reals",):
            return "double"
        elif self.basic_superset[(name,)] == ("literals",):
            return "std::string"
        return "unknown"                #pragma:nocover

    def _set_args(self,names):
        ans=""
        i=1
        flag=False
        for name in names:
            if flag:
                ans += ","
            flag=True
            #ans += self._index_str(name)+"& arg"+str(i)
            ans += "size_t arg"+str(i)
            i += 1
        return ans

    def _args(self,names):
        ans=""
        i=1
        flag=False
        for name in names:
            if flag:
                ans += ","
            flag=True
            ans += "arg"+str(i)
            i += 1
        return ans

