#  _________________________________________________________________________
#
#  Coopr: A COmmon Optimization Python Repository
#  Copyright (c) 2008 Sandia Corporation.
#  This software is distributed under the BSD License.
#  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
#  the U.S. Government retains certain rights in this software.
#  For more information, see the Coopr README.txt file.
#  _________________________________________________________________________


import os
import re
from coopr.opt.base import *
from coopr.opt.results import *
from coopr.opt.solver import *
import mockmip
import pyutilib.services
import pyutilib.misc
import pyutilib.common
import pyutilib.component.core


class GLPK(SystemCallSolver):
    """The GLPK LP/MIP solver
    """

    def __init__(self, **kwds):
        #
        # Call base constructor
        #
        kwds['type'] = 'glpk'
        SystemCallSolver.__init__(self, **kwds)
        #
        # Valid problem formats, and valid results for each format
        #
        self._valid_problem_formats=[ProblemFormat.mod, ProblemFormat.cpxlp, ProblemFormat.mps]
        self._valid_result_formats={}
        self._valid_result_formats[ProblemFormat.mod] = [ResultsFormat.soln]
        self._valid_result_formats[ProblemFormat.cpxlp] = [ResultsFormat.soln]
        self._valid_result_formats[ProblemFormat.mps] = [ResultsFormat.soln]

    def executable(self):
        executable = pyutilib.services.registered_executable("glpsol")
        if executable is None:
            pyutilib.component.core.PluginGlobals.env().log.error("Could not locate the 'glpsol' executable, which is required for solver %s" % self.name)
            self.enable = False
            return None
        return executable.get_path()

    def create_command_line(self,executable,problem_files):
        #
        # Define log file
        #
        if self.log_file is None:
           self.log_file = pyutilib.services.TempfileManager.create_tempfile(suffix = '.glpk.log')

        #
        # Define solution file
        #
        if self.soln_file is None:
            self.soln_file = pyutilib.services.TempfileManager.create_tempfile(suffix = '.glpk.soln')

        #
        # Define results file
        #
        if self._results_format is None or self._results_format == ResultsFormat.soln:
           self.results_file = self.soln_file
           
        #
        # Define command line
        #
        if self._timelimit is not None and self._timelimit > 0.0:
           timing = " --tmlim "+str(self._timelimit)+" "
        else:
           timing = ""
        if (self.mipgap is not None) and (self.mipgap > 0.0):
           mipgap = " --mipgap "+str(self.mipgap)+" "
        else:
           mipgap = ""            
        if self._problem_format == ProblemFormat.cpxlp:
            problem=" --cpxlp " + problem_files[0]
        elif self._problem_format == ProblemFormat.mps:
            problem=" --freemps " + problem_files[0]
        elif self._problem_format == ProblemFormat.mod:
            problem=" --math " + problem_files[0]
            for filename in problem_files[1:]:
                problem += " --data " + filename
        opt = ""
        for key in self.options:
            if isinstance(self.options[key],basestring) and ' ' in self.options[key]:
                opt += " --"+key+" \""+str(self.options[key])+"\""
            else:
                opt += " --"+key+" "+str(self.options[key])
        proc = self._timer + " " + executable + opt + " " + timing + mipgap + " --output " + self.soln_file + problem
        return pyutilib.misc.Bunch(cmd=proc, log_file=self.log_file, env=None)

    def process_logfile(self):
        """
        Process logfile
        """
        results = SolverResults()
        #
        # Initial values
        #
        #results.solver.statistics.branch_and_bound.number_of_created_subproblems=0
        #results.solver.statistics.branch_and_bound.number_of_bounded_subproblems=0
        soln = results.solution.add()
        soln.objective['f']=float('inf')
        #
        # Process logfile
        #
        OUTPUT = open(self.log_file)
        output = "".join(OUTPUT.readlines())
        OUTPUT.close()
        #
        # Parse logfile lines
        #
        for line in output.split("\n"):
          tokens = re.split('[ \t]+',line.strip())
          if len(tokens) > 4 and tokens[1] == "objval":
             soln.objective['f'] = tokens[3]
          elif len(tokens) > 3 and tokens[0] == "Objective" and tokens[1] == "value":
             soln.objective['f'] = tokens[3]
          elif len(tokens) > 4 and tokens[0] == "!" and tokens[2] == "objval":
             soln.objective['f'] = tokens[4]
          elif len(tokens) > 4 and tokens[0] == "+" and tokens[2] == "objval":
             soln.objective['f'] = tokens[4]
          elif len(tokens) > 4 and tokens[0] == "*" and tokens[2] == "objval":
             soln.objective['f'] = tokens[4]
          elif len(tokens) > 4 and tokens[0] == "+" and tokens[2] == "mip" and tokens[4] == "not":
             soln.objective['f'] = "unknown"
             results.problem.lower_bound = tokens[8]
          elif len(tokens) > 4 and tokens[0] == "+" and tokens[1] == "mip" and tokens[4] == "not":
             soln.objective['f'] = "unknown"
             results.problem.lower_bound = tokens[7]
          elif len(tokens) > 4 and tokens[0] == "+" and tokens[2] == "mip" and tokens[4] != "not":
             soln.objective['f'] = tokens[4]
             if tokens[6] != "tree":
                results.problem.lower_bound = tokens[6]
          elif len(tokens) > 4 and tokens[0] == "+" and tokens[1] == "mip" and tokens[4] != "not":
             soln.objective['f'] = tokens[3]
             results.problem.lower_bound = tokens[5]
          elif len(tokens) == 6 and tokens[0] == "OPTIMAL" and tokens[1] == "SOLUTION" and tokens[5] == "PRESOLVER":
             results.solver.statistics.branch_and_bound.number_of_created_subproblems = 0
             results.solver.statistics.branch_and_bound.number_of_bounded_subproblems = 0
             soln.status = SolutionStatus.optimal
          elif len(tokens) == 7 and tokens[1] == "OPTIMAL" and tokens[2] == "SOLUTION" and tokens[6] == "PRESOLVER":
             results.solver.statistics.branch_and_bound.number_of_created_subproblems = 0
             results.solver.statistics.branch_and_bound.number_of_bounded_subproblems = 0
             soln.status = SolutionStatus.optimal
          elif len(tokens) > 10 and tokens[0] == "+" and tokens[8] == "empty":
             results.solver.statistics.branch_and_bound.number_of_created_subproblems = tokens[11][:-1]
             results.solver.statistics.branch_and_bound.number_of_bounded_subproblems = tokens[11][:-1]
          elif len(tokens) > 9 and tokens[0] == "+" and tokens[7] == "empty":
             results.solver.statistics.branch_and_bound.number_of_created_subproblems = tokens[10][:-1]
             results.solver.statistics.branch_and_bound.number_of_bounded_subproblems = tokens[10][:-1]
          elif len(tokens) == 2 and tokens[0] == "sys":
             results.solver.system_time=tokens[1]
          elif len(tokens) == 2 and tokens[0] == "user":
             results.solver.user_time=tokens[1]
          elif len(tokens) > 2 and tokens[0] == "OPTIMAL" and tokens[1] == "SOLUTION":
             soln.status = SolutionStatus.optimal
          elif len(tokens) > 2 and tokens[0] == "INTEGER" and tokens[1] == "OPTIMAL":
             soln.status = SolutionStatus.optimal
             results.solver.statistics.branch_and_bound.number_of_created_subproblems = 0
             results.solver.statistics.branch_and_bound.number_of_bounded_subproblems = 0
          elif len(tokens) > 2 and tokens[0] == "TIME" and tokens[2] == "EXCEEDED;":
             soln.status = SolutionStatus.stoppedByLimit
        if results.problem.upper_bound == "inf":
           results.problem.upper_bound = 'Infinity'
        if results.problem.lower_bound == "-inf":
           results.problem.lower_bound = "-Infinity"
        try:
            val = results.problem.upper_bound
            tmp = eval(val.strip())
            results.problem.upper_bound = str(tmp)
        except:
            pass
        try:
            val = results.problem.lower_bound
            tmp = eval(val.strip())
            results.problem.lower_bound = str(tmp)
        except:
            pass
        try:
            val = soln.objective['f'].value
            tmp = eval(val.strip())
            soln.objective['f'] = str(tmp)
        except:
            pass
        if soln.status is SolutionStatus.optimal:
           soln.gap=0.0
        elif soln.status is SolutionStatus.stoppedByLimit:
           soln.gap = "Infinity" # until proven otherwise
           if "lower_bound" in dir(results.problem):
              if results.problem.lower_bound is "-Infinity":
                 soln.gap="Infinity"
              elif not results.problem.lower_bound is None:
                 if "upper_bound" not in dir(results.problem):
                    gap="Infinity"
                 elif results.problem.upper_bound is None:
                    gap="Infinity"
                 else:
                    soln.gap=eval(soln.objective['f']) - eval(results.problem.lower_bound)
           elif "upper_bound" in dir(results.problem):
              if results.problem.upper_bound is "Infinity":
                 soln.gap="Infinity"
              elif not results.problem.upper_bound is None:
                 soln.gap=eval(results.problem.upper_bound) - eval(soln.objective['f'])
        if results.solver.status is SolverStatus.error:
           results.solution.delete(0)
        return results

    def process_soln_file(self,results):

        # the only suffixes that we extract from GLPK are
        # constraint duals. scan through the solver suffix
        # list and throw an exception if the user has 
        # specified any others.
        extract_duals = False
        for suffix in self.suffixes:
            flag=False
            if re.match(suffix,"dual"):
                extract_duals = True
                flag=True
            if not flag:
                raise RuntimeError,"***GLPK solver plugin cannot extract solution suffix="+suffix

        lp_solution = True # if false, we're dealing with a MIP!
        if not os.path.exists(self.soln_file):
           return
        soln = results.solution(0)
        INPUT = open(self.soln_file,"r")
        
        state = 0 # 0=initial header, 1=constraints, 2=variables, -1=done
        
        results.problem.number_of_objectives = 1
        
        number_of_constraints_read = 0 # for validation of the total count read and the order
        number_of_variables_read = 0
        active_constraint_name = "" # constraint names and their value/bounds can be split across multiple lines
        active_variable_name = "" # variable names and their value/bounds can be split across multiple lines
        
        for line in INPUT:
          tokens = re.split('[ \t]+',line.strip())

          if (len(tokens) == 1) and (len(tokens[0]) == 0):
             pass
          elif state == 0:
             #
             # Processing initial header
             #
             if len(tokens) == 2 and tokens[0] == "Problem:":
                # the problem name may be absent, in which case the "Problem:" line will be skipped.
                results.problem.name = tokens[1]
             elif len(tokens) == 2 and tokens[0] == "Rows:":
                results.problem.number_of_constraints = eval(tokens[1])
             elif len(tokens) == 2 and tokens[0] == "Columns:":
                lp_solution = True
                results.problem.number_of_variables = eval(tokens[1])
             elif len(tokens) > 2 and tokens[0] == "Columns:":
                lp_solution = False
                results.problem.number_of_variables = eval(tokens[1])                
             elif len(tokens) == 2 and tokens[0] == "Non-zeros:":
                results.problem.number_of_nonzeros = eval(tokens[1])
             elif len(tokens) >= 2 and tokens[0] == "Status:":
                if tokens[1] == "OPTIMAL":
                   soln.status = SolutionStatus.optimal
                elif len(tokens) == 3 and tokens[1] == "INTEGER" and tokens[2] == "NON-OPTIMAL":
                   soln.status = SolutionStatus.bestSoFar
                elif len(tokens) == 3 and tokens[1] == "INTEGER" and tokens[2] == "OPTIMAL":
                   soln.status = SolutionStatus.optimal
                elif len(tokens) == 3 and tokens[1] == "INTEGER" and tokens[2] == "UNDEFINED":
                   soln.status = SolutionStatus.stoppedByLimit
                else:
                   print "GLPK WARNING: unknown status: "+" ".join(tokens[1:])
             elif len(tokens) >= 2 and tokens[0] == "Objective:":
                if tokens[4] == "(MINimum)":
                   results.problem.sense = ProblemSense.minimize
                else:
                   results.problem.sense = ProblemSense.maximize
                soln.objective['f']=tokens[3]
                if soln.status is SolutionStatus.optimal:
                   if tokens[4] == "(MINimum)":
                        results.problem.lower_bound = soln.objective['f'].value
                        if "upper_bound" in dir(results.problem):
                            del results.problem.upper_bound
                   else:
                        results.problem.upper_bound = soln.objective['f'].value
                        if "lower_bound" in dir(results.problem):
                            del results.problem.lower_bound
                # the objective is the last entry in the problem section - move on to constraints.
                state = 1

          elif state == 1:
             #
             # Process Constraint Info
             #
             
             if (len(tokens) == 2) and (len(active_constraint_name) == 0):

                number_of_constraints_read = number_of_constraints_read + 1
                active_constraint_name = tokens[1].strip()
                index = eval(tokens[0].strip())

                # sanity check - the indices should be in sequence.
                if index != number_of_constraints_read:
                   raise ValueError,"***ERROR: Unexpected constraint index encountered on line="+line+"; expected value="+str(number_of_constraints_read)+"; actual value="+str(index)

             else:

                index = None
                activity = None
                lower_bound = None
                upper_bound = None
                marginal = None

                # extract the field names and process accordingly. there
                # is some wasted processing w.r.t. single versus double-line
                # entries, but it's not significant enough to worry about.                
                 
                index_string = line[0:6].strip()
                name_string = line[7:19].strip()
                activity_string = line[23:36].strip()
                lower_bound_string = line[37:50].strip()
                upper_bound_string = line[51:64].strip()

                state_string = None                
                marginal_string = None

                # skip any headers
                if (index_string == "------") or (index_string == "No."):
                   continue

                if len(index_string) > 0:
                   index = eval(index_string)               

                if lp_solution is True:
                   state_string = line[20:22].strip()
                   marginal_string = line[65:78].strip()
                   if (activity_string != "< eps") and (len(activity_string) > 0):
                      activity = eval(activity_string)
                   else:
                      activity = 0.0
                   if (lower_bound_string != "< eps") and (len(lower_bound_string) > 0):
                      lower_bound = eval(lower_bound_string)
                   else:
                      lower_bound = 0.0
                   if state_string != "NS":                   
                      if (upper_bound_string != "< eps") and (len(upper_bound_string) > 0):
                         upper_bound = eval(upper_bound_string)
                      else:
                         upper_bound = 0.0
                   if (marginal_string != "< eps") and (len(marginal_string) > 0):
                      marginal = eval(marginal_string)
                   else:
                      marginal = 0.0

                else:
                    # no constraint-related attributes/values are extracted currently for MIPs.
                    pass
                
                constraint_name = None
                if len(active_constraint_name) > 0:
                   # if there is an active constraint name, the identifier was
                   # too long for everything to be on a single line; the second
                   # line contains all of the value information.                    
                   constraint_name = active_constraint_name
                   active_constraint_name = ""
                else:
                   # everything is on a single line.
                   constraint_name = name_string
                   number_of_constraints_read = number_of_constraints_read + 1 
                   # sanity check - the indices should be in sequence.
                   if index != number_of_constraints_read:
                      raise ValueError,"***ERROR: Unexpected constraint index encountered on line="+line+"; expected value="+str(number_of_constraints_read)+"; actual value="+str(index)
   
                if lp_solution is True:
                   # GLPK doesn't report slacks directly.
                   constraint_dual = activity
                   if state_string == "B":
                      constraint_dual = 0.0
                   elif (state_string == "NS") or (state_string == "NL") or (state_string == "NU"):
                      constraint_dual = marginal
                   else:
                      raise ValueError, "Unknown status="+tokens[0]+" encountered for constraint="+active_constraint_name+" in line="+line+" of solution file="+self.soln_file

                   if extract_duals is True:
                      soln.constraint[constraint_name].dual = constraint_dual
                  
                else:
                   # there isn't anything interesting to do with constraints in the MIP case.
                   pass

                # if all of the constraints have been read, exit.
                if number_of_constraints_read == results.problem.number_of_constraints:
                   state = 2
                      
          elif state == 2:
             #
             # Process Variable Info
             #

             if (len(tokens) == 2) and (len(active_variable_name) == 0):
                 
                # in the case of name over-flow, there are only two tokens
                # on the first of two lines for the variable entry.
                number_of_variables_read = number_of_variables_read + 1
                active_variable_name = tokens[1].strip()
                index = eval(tokens[0].strip())

                # sanity check - the indices should be in sequence.
                if index != number_of_variables_read:
                   raise ValueError,"***ERROR: Unexpected variable index encountered on line="+line+"; expected value="+str(number_of_variables_read)+"; actual value="+str(index)                    
               
             else:
                 
                index = None
                activity = None
                lower_bound = None
                upper_bound = None
                marginal = None

                # extract the field names and process accordingly. there
                # is some wasted processing w.r.t. single versus double-line
                # entries, but it's not significant enough to worry about.

                index_string = line[0:6].strip()
                name_string = line[7:19].strip()
                activity_string = line[23:36].strip()
                lower_bound_string = line[37:50].strip()
                upper_bound_string = line[51:64].strip()

                state_string = None
                marginal_string = None

                # skip any headers
                if (index_string == "------") or (index_string == "No."):
                   continue

                if len(index_string) > 0:
                   index = eval(index_string)

                if lp_solution is True:
                   state_string = line[20:22].strip()
                   marginal_string = line[65:78].strip()

                   if (activity_string != "< eps") and (len(activity_string) > 0):
                      activity = eval(activity_string)
                   else:
                      activity = 0.0
                   if (lower_bound_string != "< eps") and (len(lower_bound_string) > 0):
                      lower_bound = eval(lower_bound_string)
                   else:
                      lower_bound = 0.0
                   if state_string != "NS":                   
                      if (upper_bound_string != "< eps") and (len(upper_bound_string) > 0):
                         upper_bound = eval(upper_bound_string)
                      else:
                         upper_bound = 0.0
                   if (marginal_string != "< eps") and (len(marginal_string) > 0):
                      marginal = eval(marginal_string)
                   else:
                      marginal = 0.0

                else:

                   if (activity_string != "< eps") and (len(activity_string) > 0):
                      activity = eval(activity_string)
                   else:
                      activity = 0.0                    

                variable_name = None
                if len(active_variable_name) > 0:
                   # if there is an active variable name, the identifier was
                   # too long for everything to be on a single line; the second
                   # line contains all of the value information.
                   variable_name = active_variable_name
                   active_variable_name = ""
                else:
                   # everything is on a single line.
                   variable_name = name_string
                   number_of_variables_read = number_of_variables_read + 1 
                   # sanity check - the indices should be in sequence.
                   if index != number_of_variables_read:
                      raise ValueError,"***ERROR: Unexpected variable index encountered on line="+line+"; expected value="+str(number_of_variables_read)+"; actual value="+str(index)

                if lp_solution is True:
                   # the "activity" column always specifies the variable value.
                   # embedding the if-then-else to validate the basis status.
                   # we are currently ignoring all bound-related information.
                   variable_value = None
                   if state_string == "B":
                      variable_value = activity
                   elif (state_string == "NL") or (state_string == "NS") or (state_string == "NU"):
                      variable_value = activity
                   else:
                      raise ValueError, "Unknown status="+state_string+" encountered for variable="+active_variable_name+" in line="+line+" of solution file="+self.soln_file
               
                   soln.variable[variable_name].value = variable_value
                else:
                   soln.variable[variable_name].value = activity
                
             # if all of the variables have been read, exit.
             if number_of_variables_read == results.problem.number_of_variables:
                state = -1
             
          if state==-1:
             break
         
        INPUT.close()

class MockGLPK(GLPK,mockmip.MockMIP):
    """A Mock GLPK solver used for testing
    """

    def __init__(self, **kwds):
        try:
           GLPK.__init__(self, **kwds)
        except pyutilib.common.ApplicationError: #pragma:nocover
           pass                        #pragma:nocover
        mockmip.MockMIP.__init__(self,"glpk")

    def available(self, exception_flag=True):
        return GLPK.available(self,exception_flag)

    def create_command_line(self,executable,problem_files):
        command = GLPK.create_command_line(self,executable,problem_files)
        mockmip.MockMIP.create_command_line(self,executable,problem_files)
        return command

    def executable(self):
        return mockmip.MockMIP.executable(self)

    def _execute_command(self,cmd):
        return mockmip.MockMIP._execute_command(self,cmd)

    def _convert_problem(self,args,pformat,valid_pformats):
        if pformat in [ProblemFormat.mps,ProblemFormat.cpxlp]:
           return (args,pformat,None)
        else:
           return (args,ProblemFormat.cpxlp,None)


pyutilib.services.register_executable(name="glpsol")
SolverRegistration("glpk", GLPK)
SolverRegistration("_mock_glpk", MockGLPK)
