#  _________________________________________________________________________
#
#  Coopr: A COmmon Optimization Python Repository
#  Copyright (c) 2008 Sandia Corporation.
#  This software is distributed under the BSD License.
#  Under the terms of Contract DE-AC04-94AL85000 with Sandia Corporation,
#  the U.S. Government retains certain rights in this software.
#  For more information, see the Coopr README.txt file.
#  _________________________________________________________________________

__all__ = ['CBC', 'MockCBC']

import os
import re
from coopr.opt.base import *
from coopr.opt.results import *
from coopr.opt.solver import *
import pyutilib.services
import pyutilib.misc
import pyutilib.common
import pyutilib.component.core
import pyutilib.subprocess
import mockmip

cbc_compiled_with_asl=None

def configure():
    global cbc_compiled_with_asl
    if cbc_compiled_with_asl is None:
        cbc = CBC()
        if not cbc.executable() is None:
            results = pyutilib.subprocess.run(cbc.executable()+" dummy -AMPL")
            if not cbc_compiled_with_asl is None:
                raise IOError
            cbc_compiled_with_asl = not ('No match for AMPL' in results[1])
        else:
            cbc_compiled_with_asl = False


class CBC(SystemCallSolver):
    """The CBC LP/MIP solver
    """

    def __init__(self, **kwds):
        #
        # Call base constructor
        #
        kwds['type'] = 'cbc'
        SystemCallSolver.__init__(self, **kwds)

        #
        # Set up valid problem formats and valid results for each problem format
        #
#        self._valid_problem_formats=[ProblemFormat.nl, ProblemFormat.cpxlp, ProblemFormat.mps]
        self._valid_problem_formats=[ProblemFormat.cpxlp, ProblemFormat.mps]        
        if cbc_compiled_with_asl:
            self._valid_problem_formats.append(ProblemFormat.nl)
        self._valid_result_formats={}
        self._valid_result_formats[ProblemFormat.cpxlp] = [ResultsFormat.soln]
        if cbc_compiled_with_asl:
            self._valid_result_formats[ProblemFormat.nl] = [ResultsFormat.sol]        
        self._valid_result_formats[ProblemFormat.mps] = [ResultsFormat.soln]

    def _presolve(self, *args, **kwds):

       # establish real "default" problem and results formats. these may be
       # over-ridden in the base class solve (via keywords), but we should
       # have real values by the time we're presolving.
       if self._problem_format is None:

          if self._valid_problem_formats[0] is ProblemFormat.nl:
             self._problem_format = ProblemFormat.nl
          else:
             self._problem_format = ProblemFormat.cpxlp

       # in CBC, the results format is defined by the problem format;
       # you can't vary them independently.
       if cbc_compiled_with_asl and self._problem_format is ProblemFormat.nl:
          self._results_format = ResultsFormat.sol
       else:
          # this really means "CBC-specific" format - not drawing from the
          # log file itself (although additional information is contained there).
          self._results_format = ResultsFormat.soln

       # let the base class handle any remaining keywords/actions. 
       SystemCallSolver._presolve(self, *args, **kwds)        

    def executable(self):
        executable = pyutilib.services.registered_executable("cbc")
        if executable is None:
            pyutilib.component.core.PluginGlobals.env().log.error("Could not locate the 'cbc' executable, which is required for solver %s" % self.name)
            self.enable = False
            return None
        return executable.get_path()

    def create_command_line(self,executable,problem_files):
        #
        # Define the log file 
        #
        if self.log_file is None:
           self.log_file = pyutilib.services.TempfileManager.create_tempfile(suffix=".cbc.log")

        #
        # Define the solution file
        #

        # the prefix of the problem filename is required because CBC has a specific
        # and automatic convention for generating the output solution filename.
        # the extracted prefix is the same name as the input filename, e.g., minus
        # the ".lp" extension.
        problem_filename_prefix = problem_files[0]
        if '.' in problem_filename_prefix:
            tmp = problem_filename_prefix.split('.')
            if len(tmp) > 2:
                problem_filename_prefix = '.'.join(tmp[:-1])
            else:
                problem_filename_prefix = tmp[0]

        #if not cbc_compiled_with_asl:
            #self._results_format = ResultsFormat.soln
            #self.results_reader = None
        if self._results_format is ResultsFormat.sol:
           self.soln_file = problem_filename_prefix+".sol"
        else:
           self.soln_file = problem_filename_prefix+".soln"

        #
        # Define the results file
        #
        # results in CBC are split across the log file (solver statistics) and
        # the solution file (solutions!)
        self.results_file = self.soln_file
        
        #
        # Define command line
        #
        if (self.mipgap is not None):
           raise ValueError, "The mipgap parameter is currently not being processed by CBC solver plugin"
        
        if self._problem_format == ProblemFormat.nl:
            if self._timelimit is not None and self._timelimit > 0.0:
                timing = " sec="+str(self._timelimit)
            else:
                timing = ""
            if "debug" in self.options:
                opt = ""
            else:
                opt = " log=5"
            for key in self.options:
                if isinstance(self.options[key],basestring) and ' ' in self.options[key]:
                    opt += " "+key+"=\""+str(self.options[key])+"\""
                else:
                    opt += " "+key+"="+str(self.options[key])
            self._nl_options = " ".join(self.options) + timing+ " stat=1 "+opt+" printingOptions=all"
            proc = self._timer + " " + executable + " " + problem_files[0] + " -AMPL "+self._nl_options
        else:
            if self._timelimit is not None and self._timelimit > 0.0:
                timing = " -sec "+str(self._timelimit)+" "
            else:
                timing = ""
            if "debug" in self.options:
                opt = ""
            else:
                opt = " -log 5"
            for key in self.options:
                if isinstance(self.options[key],basestring) and ' ' in self.options[key]:
                    opt += " -"+key+" \""+str(self.options[key])+"\""
                else:
                    opt += " -"+key+" "+str(self.options[key])
            proc = self._timer + " " + executable + opt + " -printingOptions all "+timing+ " -import "+problem_files[0]+ " -import -stat 1 -solve -solu " + self.soln_file
        return pyutilib.misc.Bunch(cmd=proc, log_file=self.log_file, env=None)

    def process_logfile(self):
        """
        Process logfile
        """
        results = SolverResults()
        #
        # Initial values
        #
        #results.solver.statistics.branch_and_bound.number_of_created_subproblems=0
        #results.solver.statistics.branch_and_bound.number_of_bounded_subproblems=0
        soln = results.solution.add()
        soln.objective['f'].value=float('inf')
        #
        # Process logfile
        #
        OUTPUT = open(self.log_file)
        output = "".join(OUTPUT.readlines())
        OUTPUT.close()
        #
        # Parse logfile lines
        #
        results.problem.sense = ProblemSense.minimize
        sense=1
        results.problem.name = None
        for line in output.split("\n"):
          tokens = re.split('[ \t]+',line.strip())
          #print "LINE:", line
          if len(tokens) == 10 and tokens[0] == "Current" and tokens[1] == "default" and tokens[2] == "(if" and results.problem.name is None:
             results.problem.name = tokens[-1]
             if '.' in results.problem.name:
                parts = results.problem.name.split('.')
                if len(parts) > 2:
                    results.problem.name = '.'.join(parts[:-1])
                else:
                    results.problem.name = results.problem.name.split('.')[0]
             if '/' in results.problem.name:
                results.problem.name = results.problem.name.split('/')[-1]
             if '\\' in results.problem.name:
                results.problem.name = results.problem.name.split('\\')[-1]
          if len(tokens) ==11 and tokens[0] == "Presolve" and tokens[3] == "rows,":
             results.problem.number_of_variables = eval(tokens[4])-eval(tokens[5][1:-1])
             results.problem.number_of_constraints = eval(tokens[1])-eval(tokens[2][1:-1])
             results.problem.number_of_nonzeros = eval(tokens[8])-eval(tokens[9][1:-1])
             results.problem.number_of_objectives = "1"
          if len(tokens) >=9 and tokens[0] == "Problem" and tokens[2] == "has":
             results.problem.number_of_variables = tokens[5]
             results.problem.number_of_constraints = tokens[3]
             results.problem.number_of_nonzeros = tokens[8]
             results.problem.number_of_objectives = "1"
          if len(tokens) == 5 and tokens[3] == "NAME":
             results.problem.name = tokens[4]
          if " ".join(tokens) == '### WARNING: CoinLpIO::readLp(): Maximization problem reformulated as minimization':
             results.problem.sense = ProblemSense.maximize
             sense = -1
          if len(tokens) > 3 and tokens[0] == "Presolve" and tokens[6] == "infeasible":
             soln.status = SolutionStatus.infeasible
             soln.objective['f'].value=None
          if len(tokens) > 3 and tokens[0] == "Optimal" and tokens[1] == "objective":
             soln.status = SolutionStatus.optimal
             soln.objective['f'].value=tokens[2]
          if len(tokens) > 6 and tokens[4] == "best" and tokens[5] == "objective":
             soln.objective['f'].value=tokens[6]
          if len(tokens) > 9 and tokens[7] == "(best" and tokens[8] == "possible":
             results.problem.lower_bound=tokens[9]
             results.problem.lower_bound = results.problem.lower_bound.split(")")[0]
          if len(tokens) > 12 and tokens[10] == "best" and tokens[11] == "possible":
             results.problem.lower_bound=tokens[12]
          if len(tokens) > 3 and tokens[0] == "Result" and tokens[2] == "Finished":
             soln.status = SolutionStatus.optimal
             soln.objective['f'].value=tokens[4]
          if len(tokens) > 10 and tokens[4] == "time" and tokens[9] == "nodes":
             results.solver.statistics.branch_and_bound.number_of_created_subproblems=tokens[8]
             results.solver.statistics.branch_and_bound.number_of_bounded_subproblems=tokens[8]
             if eval(results.solver.statistics.branch_and_bound.number_of_bounded_subproblems) > 0:
                soln.objective['f'].value=tokens[6]
          if len(tokens) == 5 and tokens[1] == "Exiting" and tokens[4] == "time":
             soln.status = SolutionStatus.stoppedByLimit
          if len(tokens) > 8 and tokens[7] == "nodes":
             results.solver.statistics.branch_and_bound.number_of_created_subproblems=tokens[6]
             results.solver.statistics.branch_and_bound.number_of_bounded_subproblems=tokens[6]
          if len(tokens) == 2 and tokens[0] == "sys":
             results.solver.system_time=tokens[1]
          if len(tokens) == 2 and tokens[0] == "user":
             results.solver.user_time=tokens[1]
          results.solver.user_time=-1.0

        if soln.objective['f'].value == "1e+50":
           if sense == 1:
              soln.objective['f'].value=float('inf')
           else:
              soln.objective['f'].value=float('-inf')
        elif not soln.objective['f'].value is None:
           soln.objective['f'].value = eval(soln.objective['f'].value)*sense
        if soln.status is SolutionStatus.optimal:
           soln.gap=0.0
           if results.problem.sense == ProblemSense.minimize:
                results.problem.lower_bound = soln.objective['f'].value
                if "upper_bound" in dir(results.problem):
                    del results.problem.upper_bound
           else:
                results.problem.upper_bound = soln.objective['f'].value
                if "lower_bound" in dir(results.problem):
                    del results.problem.lower_bound
        if results.solver.status is SolverStatus.error:
           results.solution.delete(0)
        return results

    def process_soln_file(self,results):

        # the only suffixes that we extract from CBC are
        # constraint duals and variable reduced-costs. scan 
        # through the solver suffix list and throw an 
        # exception if the user has specified any others.
        extract_duals = False
        extract_reduced_costs = False
        for suffix in self.suffixes:
           flag=False
           if re.match(suffix, "dual"):
              extract_duals = True
              flag=True
           if re.match(suffix, "rc"):
              extract_reduced_costs = True
              flag=True
           if not flag:
              raise RuntimeError,"***CBC solver plugin cannot extract solution suffix="+suffix

        # if dealing with SOL format files, we've already read
        # this via the base class reader functionality.
        if self._results_format is ResultsFormat.sol:
           return

        # otherwise, go with the native CBC solution format.
        solution = results.solution(0)
        if solution.status is SolutionStatus.infeasible:
            # NOTE: CBC _does_ print a solution file.  However, I'm not
            # sure how to interpret it yet.
            return
        results.problem.number_of_objectives=1

        processing_constraints=None # None means header, True means constraints, False means variables.
        INPUT = open(self.soln_file,"r")
        for line in INPUT:
          tokens = re.split('[ \t]+',line.strip())
          #print "LINE",line,len(tokens)
          if tokens[0] in ("Optimal", "Status"):
             # TBD - this logic isn't correct/complete. flush out as necessary.              
             continue
          if tokens[0] == "0": # indicates section start.
             if processing_constraints is None:
                processing_constraints = True
             elif processing_constraints is True:
                processing_constraints = False
             else:
                raise RuntimeError, "CBC encountered unexpected line=("+line.strip()+") in solution file="+self.soln_file+"; constraint and variable sections already processed!"

          if (processing_constraints is True) and (extract_duals is True):
             constraint = tokens[1]
             constraint_ax = eval(tokens[2]) # CBC reports the constraint row times the solution vector - not the slack.
             constraint_dual = eval(tokens[3])

             solution.constraint[constraint].dual = constraint_dual

          elif processing_constraints is False:
             variable_name = tokens[1]
             variable_value = eval(tokens[2])
             solution.variable[variable_name].value = variable_value

             if extract_reduced_costs is True:
                variable_reduced_cost = eval(tokens[3]) # currently ignored.
                solution.variable[variable_name].rc = variable_reduced_cost
          else:
             raise RuntimeError, "CBC encountered unexpected line=("+line.strip()+") in solution file="+self.soln_file+"; expecting header, but found data!"
         
        INPUT.close()


class MockCBC(CBC,mockmip.MockMIP):
    """A Mock CBC solver used for testing
    """

    def __init__(self, **kwds):
        try:
           CBC.__init__(self,**kwds)
        except pyutilib.common.ApplicationError: #pragma:nocover
           pass                        #pragma:nocover
        mockmip.MockMIP.__init__(self,"cbc")

    def available(self, exception_flag=True):
        return CBC.available(self,exception_flag)

    def create_command_line(self,executable,problem_files):
        command = CBC.create_command_line(self,executable,problem_files)
        mockmip.MockMIP.create_command_line(self,executable,problem_files)
        return command

    def executable(self):
        return mockmip.MockMIP.executable(self)

    def _execute_command(self,cmd):
        return mockmip.MockMIP._execute_command(self,cmd)

    def _convert_problem(self,args,pformat,valid_pformats):
        if pformat in [ProblemFormat.mps, ProblemFormat.cpxlp, ProblemFormat.nl]:
           return (args, pformat, None)
        else:
           return (args, ProblemFormat.mps, None)


pyutilib.services.register_executable(name="cbc")
SolverRegistration("cbc", CBC)
SolverRegistration("_mock_cbc", MockCBC)
