# read data file as character vector
data   <- scan(file="nonlinear_ok_sin.out", what="character")
#
# set some line types
solid    <- 1
dashed   <- 2
dotted   <- 3
dotdash  <- 4
longdash <- 5
twodast  <- 6
#
# minimum allowable value for x4(t)
x4_min <- .25
#
# convert to a character matrix
len    <- length(data)
nc     <- 8
#
# separate from the header line
header <- data[1 : nc]
data   <- data[(nc+1) : len]
#
# convert data to double precision
data   <- as.double(data)
#
# convert to a matrix
nr     <- (len - 1)  / nc
data   <- matrix(data, nrow=nr, ncol=nc, byrow=TRUE)
#
# plot true values with dotted line
x2_true <- data[, "x2_true" == header]
x4_true <- data[, "x4_true" == header]
plot(
	x = x2_true, 
	y = x4_true, 
	type = "l", 
	ylim = c(0, 2.5),
	main = "Nonlinear Kalman-Bucy Smoother",
	xlab = "x2", 
	ylab = "x4", 
	lty=dotted
)
#
# plot lower bound with a straight line
x4_low   <- 1 - sin( x2_true ) + x4_min
lines(x = x2_true, y = x4_low, type = "l", lty=solid)
#
# plot constrained estimate with longdash line
x2_con   <- data[, "x2_con" == header]
x4_con   <- data[, "x4_con" == header]
lines(x = x2_con, y = x4_con, type = "l", lty=longdash)
#
# plot unconstrained estimate with dashed line
x2_free <- data[, "x2_free" == header]
x4_free <- data[, "x4_free" == header]
lines(x = x2_free, y = x4_free, type = "l", lty=dashed)
#
# add legend to plot
xleg    <- .3 
yleg    <- 2.5
legend (
	x = xleg , 
	y = yleg, 
	legend = c("true",     "con",   "free",   "bound"), 
	lty    = c(dotted,  longdash,   dashed,     solid) 
)
#
# save the plot in encapsulated postscript
savePlot(filename = "nonlinear_ok_sin", type = "eps");
